/*      br_chash.c
 *
 *	Hash tables for encrypting bridge: hash(addr) == encryption context
 *      By Torrey Hoffman
 *
 *      br_chash.c was originally based on:
 *        "br_filter.c: Mac + Protocol filtering database", which
 *        is an add-on patch for the kernel ethernet bridge code.
 *
 *	br_filter.c authors:
 *	Christian Welzel		<Sir_Gawain@gmx.de>
 *      Arne Fitzenreiter               <arne@fitzenreiter.de>
 *
 *	This program is free software; you can redistribute it and/or
 *	modify it under the terms of the GNU General Public License
 *	as published by the Free Software Foundation; either version
 *	2 of the License, or (at your option) any later version.
 */

#include <linux/kernel.h>
#include <linux/spinlock.h>
#include <linux/if_bridge.h>
#include <asm/atomic.h>
#include <asm/uaccess.h>
#include "br_private.h"

// calculates the hash for a mac
static __inline__ int br_mac_hash(unsigned char *mac)
{
	unsigned long x;

	x = mac[0];
	x = (x << 2) ^ mac[1];
	x = (x << 2) ^ mac[2];
	x = (x << 2) ^ mac[3];
	x = (x << 2) ^ mac[4];
	x = (x << 2) ^ mac[5];

	x ^= x >> 8;

	return x & (BR_HASH_SIZE - 1);
}

// insert entry into encrypt hashtable list
static __inline__ void __encrypt_macs_link(struct net_bridge *br,
					   struct net_bridge_chash_entry *ent,
					   int hash)
{
	ent->next_hash = br->encrypt_macs[hash];
	if (ent->next_hash != NULL)
		ent->next_hash->pprev_hash = &ent->next_hash;
	br->encrypt_macs[hash] = ent;
	ent->pprev_hash = &br->encrypt_macs[hash];
}

// insert entry into decrypt hashtable list
static __inline__ void __decrypt_macs_link(struct net_bridge *br,
					   struct net_bridge_chash_entry *ent,
					   int hash)
{
	ent->next_hash = br->decrypt_macs[hash];
	if (ent->next_hash != NULL)
		ent->next_hash->pprev_hash = &ent->next_hash;
	br->decrypt_macs[hash] = ent;
	ent->pprev_hash = &br->decrypt_macs[hash];
}

// remove entry from hashtable list
static __inline__ void __crypt_macs_unlink(struct net_bridge_chash_entry *ent)
{
	*(ent->pprev_hash) = ent->next_hash;
	if (ent->next_hash != NULL)
		ent->next_hash->pprev_hash = ent->pprev_hash;
	ent->next_hash = NULL;
	ent->pprev_hash = NULL;
}

// tests if entry is unused -> deletes it
void br_chash_delete(struct net_bridge_chash_entry *ent)
{
	if (atomic_dec_and_test(&ent->use_count)) {
		br_aes_deletecontext(ent->context);
		ent->context = NULL;
		kfree(ent);
	}
}

// returns pointer to encryption context if addr is on encryption list, otherwise NULL
aes_ctx * br_chash_get_enc(struct net_bridge *br, unsigned char *addr)
{
	struct net_bridge_chash_entry *ent;

	read_lock_bh(&br->crypt_macs_lock);
	ent = br->encrypt_macs[br_mac_hash(addr)];
	while (ent != NULL) {
		if (!memcmp(ent->addr.addr, addr, ETH_ALEN)) {
		        read_unlock_bh(&br->crypt_macs_lock);
		        return ent->context;
		}
		ent = ent->next_hash;
	}

	read_unlock_bh(&br->crypt_macs_lock);
	return NULL;
}

// returns pointer to decryption context if addr is on decryption list, otherwise NULL
aes_ctx * br_chash_get_dec(struct net_bridge *br, unsigned char *addr)
{
	struct net_bridge_chash_entry *ent;

	read_lock_bh(&br->crypt_macs_lock);
	ent = br->decrypt_macs[br_mac_hash(addr)];
	while (ent != NULL) {
		if (!memcmp(ent->addr.addr, addr, ETH_ALEN)) {
		        read_unlock_bh(&br->crypt_macs_lock);
		        return ent->context;
		}
		ent = ent->next_hash;
	}

	read_unlock_bh(&br->crypt_macs_lock);
	return NULL;
}

// returns all (mac, encryption context) pairs to userspace
int br_chash_get_entries(struct net_bridge *br,
			unsigned char *_buf,
			int maxnum,
			int offset)
{
       int i;
       int num;
       struct __crypthash_entry *walk;

       num = 0;
       walk = (struct __crypthash_entry *)_buf;

       read_lock_bh(&br->crypt_macs_lock);
       for (i = 0; i < BR_HASH_SIZE; i++) {
               struct net_bridge_chash_entry *f;

               f = br->encrypt_macs[i];
               while (f != NULL && num < maxnum) {
                       struct __crypthash_entry ent;
                       int err;
                       struct net_bridge_chash_entry *g;
                       struct net_bridge_chash_entry **pp;

                       if (offset) {
                               offset--;
                               f = f->next_hash;
                               continue;
                       }

                       memset(&ent, 0, sizeof(struct __crypthash_entry));
                       memcpy(ent.mac_addr, f->addr.addr, ETH_ALEN);

                       atomic_inc(&f->use_count);
                       read_unlock_bh(&br->crypt_macs_lock);
                       err = copy_to_user(walk, &ent, sizeof(struct __crypthash_entry));
		       read_lock_bh(&br->crypt_macs_lock);

                       g = f->next_hash;
                       pp = f->pprev_hash;
                       br_chash_delete(f);
  
                       if (err)
		               goto out_fault;
		       if (g == NULL && pp == NULL)
		               goto out_disappeared;
                       num++;
                       walk++;
		     
                       f = g;
               }
	}
out:
	read_unlock_bh(&br->crypt_macs_lock);
        return num;

out_disappeared:
	num = -EAGAIN;
	goto out;

out_fault:
        num = -EFAULT;
        goto out;
}

// Creates an encryption or decryption context, inserts (address, context) pair into hashtable
// returns 0 on success, error code otherwise
int br_chash_insert(struct net_bridge *br,
		     unsigned char *addr,
		     unsigned char *key,
		     unsigned int flags)
{
	struct net_bridge_chash_entry *ent;
	aes_ctx *newcontext;
	int index;
	
	index = br_mac_hash(addr);
	write_lock_bh(&br->crypt_macs_lock);

	/* do we already have an entry for this mac? */
	ent = br->encrypt_macs[index];
	while (ent != NULL) {
		if (!memcmp(ent->addr.addr, addr, ETH_ALEN)) {
			write_unlock_bh(&br->crypt_macs_lock);
			return -EINVAL;
		}
		ent = ent->next_hash;
	}
	ent = br->decrypt_macs[index];
	while (ent != NULL) {
		if (!memcmp(ent->addr.addr, addr, ETH_ALEN)) {
			write_unlock_bh(&br->crypt_macs_lock);
			return -EINVAL;
		}
		ent = ent->next_hash;
	}

	/* no entry exists, make one */
	ent = kmalloc(sizeof(*ent), GFP_ATOMIC);
	if (ent == NULL) {
		write_unlock_bh(&br->crypt_macs_lock);
		return -ENOMEM;
	}

	/* encrypt or decrypt? */
	newcontext = (flags & BR_ENCRYPT_FLAG) ? br_aes_enc_context(key) : br_aes_dec_context(key);
	if (newcontext == NULL) {
		write_unlock_bh(&br->crypt_macs_lock);
		return -ENOMEM;
	}

	memcpy(ent->addr.addr, addr, ETH_ALEN);
	ent->context = newcontext;
	atomic_set(&ent->use_count, 1);
	
	(flags & BR_ENCRYPT_FLAG) ? __encrypt_macs_link(br, ent, index) : __decrypt_macs_link(br, ent, index);
	
	write_unlock_bh(&br->crypt_macs_lock);
	return 0;
}

// finds and removes (address, context) from hashtable, stops encryption or decryption.
// returns 0 on success, error code otherwise. "flags" is unused for now.
int br_chash_remove(struct net_bridge *br, unsigned char *addr, unsigned int flags)
{
	struct net_bridge_chash_entry *f;
	int err = 0;

	write_lock_bh(&br->crypt_macs_lock);

	f = br->encrypt_macs[br_mac_hash(addr)];
	while ( f && (memcmp(f->addr.addr, addr, ETH_ALEN)) ) {
		f = f->next_hash;
	}
	if (f) {
		__crypt_macs_unlink(f);
		br_chash_delete(f);
		f = NULL;
		goto unlock_exit;
	}

	f = br->decrypt_macs[br_mac_hash(addr)];
	while ( f && (memcmp(f->addr.addr, addr, ETH_ALEN)) ) {
		f = f->next_hash;
	}
	if (f) {
		__crypt_macs_unlink(f);
		br_chash_delete(f);
		f = NULL;
		goto unlock_exit;
	}

	err = -EINVAL;

 unlock_exit:	
	write_unlock_bh(&br->crypt_macs_lock);
	return err;
}
