#ifndef _X86_64_PGALLOC_H
#define _X86_64_PGALLOC_H

#include <linux/config.h>
#include <asm/processor.h>
#include <asm/fixmap.h>
#include <linux/threads.h>
#include <asm/page.h>
#include <linux/list.h>

#define pgd_quicklist (current_cpu_data.pgd_quick)
#define pmd_quicklist (current_cpu_data.pmd_quick)
#define pte_quicklist (current_cpu_data.pte_quick)
#define pgtable_cache_size (current_cpu_data.pgtable_cache_sz)

#define pmd_populate(mm, pmd, page) set_pmd(pmd, mk_pmd(page, __pgprot(_PAGE_TABLE)))
#define pgd_populate(mm, pgd, pmd) \
		set_pgd(pgd, __pgd(_PAGE_TABLE | __pa(pmd)))

extern __inline__ pmd_t *get_pmd_slow(void)
{
	return (pmd_t *)get_zeroed_page(GFP_KERNEL);
}

extern __inline__ void pmd_free(pmd_t *pmd)
{
	*(unsigned long *)pmd = (unsigned long) pmd_quicklist;
	pmd_quicklist = (unsigned long *) pmd;
	pgtable_cache_size++;
}

extern __inline__ void pmd_free_slow(pmd_t *pmd)
{
#ifdef CONFIG_CHECKING
	if ((unsigned long)pmd & (PAGE_SIZE-1)) 
		BUG();
#endif
	free_page((unsigned long)pmd);
}

static inline pmd_t *pmd_alloc_one_fast (struct mm_struct *mm, unsigned long addr)
{
	unsigned long *ret;

	if ((ret = (unsigned long *)pmd_quicklist) != NULL) { /* avoid likely/unlikely here */
		pmd_quicklist = (unsigned long *)(*ret);
		ret[0] = 0;
		pgtable_cache_size--;
	}
	return (pmd_t *)ret;
}

static inline pmd_t *pmd_alloc_one (struct mm_struct *mm, unsigned long addr)
{
	return (pmd_t *)get_zeroed_page(GFP_KERNEL); 
}

static inline pgd_t *get_pgd_slow(void)
{
	return (pgd_t *)get_zeroed_page(GFP_KERNEL); 
}

static inline pgd_t *pgd_alloc(struct mm_struct *mm)
{
	unsigned long *ret;

	if ((ret = pgd_quicklist) != NULL) {
		pgd_quicklist = (unsigned long *)(*ret);
		ret[0] = 0;
		pgtable_cache_size--;
	} else
		ret = (unsigned long *)get_pgd_slow();
	return (pgd_t *)ret;
}

static inline void pgd_free (pgd_t *pgd)
{
	*(unsigned long *)pgd = (unsigned long) pgd_quicklist;
	pgd_quicklist = (unsigned long *) pgd;
	pgtable_cache_size++;
}


static inline void free_pgd_slow (pgd_t *pgd)
{
#ifdef CONFIG_CHECKING
	if ((unsigned long)pgd & (PAGE_SIZE-1)) 
		BUG();
#endif
	free_page((unsigned long)pgd);
}


static inline struct page * pte_alloc_one_fast(struct mm_struct *mm,
					       unsigned long address)
{
	struct list_head * entry = pte_quicklist.next; /* FIFO */
	struct page * page = NULL;

	if (entry != &pte_quicklist) { /* don't add a likely/unlikely here */
		list_del(entry);
		page = list_entry(entry, struct page, list);
		pgtable_cache_size--;
	}
	return page;
}

static inline struct page * pte_alloc_one_fast_lifo(struct mm_struct *mm,
						    unsigned long address)
{
	struct list_head * entry = pte_quicklist.prev; /* LIFO */
	struct page * page = NULL;

	if (entry != &pte_quicklist) {
		list_del(entry);
		page = list_entry(entry, struct page, list);
		pgtable_cache_size--;
	}
	return page;
}

/* Should really implement gc for free page table pages. This could be done with 
   a reference count in struct page. */

static inline void pte_free(struct page * page)
{	
	list_add(&page->list, &pte_quicklist);
	pgtable_cache_size++;
}

static __inline__ void pte_free_slow(struct page * page)
{
	__free_page(page);
}

static inline void pte_free_via_pmd(pmd_t pmd)
{
	pte_free(virt_to_page(pte_offset(&pmd, 0)));
}

extern int do_check_pgt_cache(int, int);

/*
 * TLB flushing:
 *
 *  - flush_tlb() flushes the current mm struct TLBs
 *  - flush_tlb_all() flushes all processes TLBs
 *  - flush_tlb_mm(mm) flushes the specified mm context TLB's
 *  - flush_tlb_page(vma, vmaddr) flushes one page
 *  - flush_tlb_range(mm, start, end) flushes a range of pages
 *  - flush_tlb_pgtables(mm, start, end) flushes a range of page tables
 */

#ifndef CONFIG_SMP

#define flush_tlb() __flush_tlb()
#define flush_tlb_all() __flush_tlb_all()
#define local_flush_tlb() __flush_tlb()

static inline void flush_tlb_mm(struct mm_struct *mm)
{
	if (mm == current->active_mm)
		__flush_tlb();
}

static inline void flush_tlb_page(struct vm_area_struct *vma,
	unsigned long addr)
{
	if (vma->vm_mm == current->active_mm)
		__flush_tlb_one(addr);
}

static inline void flush_tlb_range(struct mm_struct *mm,
	unsigned long start, unsigned long end)
{
	if (mm == current->active_mm)
		__flush_tlb();
}

#else

#include <asm/smp.h>

#define local_flush_tlb() \
	__flush_tlb()

extern void flush_tlb_all(void);
extern void flush_tlb_current_task(void);
extern void flush_tlb_mm(struct mm_struct *);
extern void flush_tlb_page(struct vm_area_struct *, unsigned long);

#define flush_tlb()	flush_tlb_current_task()

static inline void flush_tlb_range(struct mm_struct * mm, unsigned long start, unsigned long end)
{
	flush_tlb_mm(mm);
}

#define TLBSTATE_OK	1
#define TLBSTATE_LAZY	2

struct tlb_state
{
	struct mm_struct *active_mm;
	int state;
};
extern struct tlb_state cpu_tlbstate[NR_CPUS];


#endif

extern inline void flush_tlb_pgtables(struct mm_struct *mm,
				      unsigned long start, unsigned long end)
{
	flush_tlb_mm(mm);
}

#endif /* _X86_64_PGALLOC_H */
