/*******************************************************************
 * This file is part of the Emulex Linux Device Driver for         *
 * Enterprise Fibre Channel Host Bus Adapters.                     *
 * Refer to the README file included with this package for         *
 * driver version and adapter support.                             *
 * Copyright (C) 2003 Emulex Corporation.                          *
 * www.emulex.com                                                  *
 *                                                                 *
 * This program is free software; you can redistribute it and/or   *
 * modify it under the terms of the GNU General Public License     *
 * as published by the Free Software Foundation; either version 2  *
 * of the License, or (at your option) any later version.          *
 *                                                                 *
 * This program is distributed in the hope that it will be useful, *
 * but WITHOUT ANY WARRANTY; without even the implied warranty of  *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the   *
 * GNU General Public License for more details, a copy of which    *
 * can be found in the file COPYING included with this package.    *
 *******************************************************************/

#include "fc_os.h"

#include "fc_hw.h"
#include "fc.h"

#include "fcdiag.h"
#include "hbaapi.h"
#include "fcfgparm.h"
#include "fcmsg.h"
#include "fc_crtn.h"   
#include "fc_ertn.h"   /* Environment - external routine definitions */

extern clock_t fc_ticks_per_second;
extern fc_dd_ctl_t   DD_CTL;
extern iCfgParam     icfgparam[];
extern int           lpfc_nethdr;
extern uint32        fcPAGESIZE;
extern uint32 fc_diag_state;
extern int    fcinstance[];

/* Routine Declaration - Local */
_local_  void fc_rscndisc_timeout(fc_dev_ctl_t *p_dev_ctl, void *l1, void *l2);
_local_  int  fc_ring_txcnt(FC_BRD_INFO *binfo, int flag);
_local_  int  fc_ring_txpcnt(FC_BRD_INFO *binfo, int flag);
/* End Routine Declaration - Local */

static uchar fcbroadcastaddr[MACADDR_LEN] = 
{ 
   0xff, 0xff, 0xff, 0xff, 0xff, 0xff
};

_static_ int fc_max_ns_retry = 3;
_static_ int fc_inq_hbeat_tmo = 60;
_static_ int fc_inq_sn_tmo = 30;
_static_ int fc_offline_stop_io = 1;
_static_ int fc_max_els_sent = 32;

#define INQ_LEN         0x100

#define RPTLUN_MIN_LEN  0x1000
#define WD6             (IOCB_WORD_SZ-2)  /* IOCB wd 6 */
#define WD7             (IOCB_WORD_SZ-1)  /* IOCB wd 7 */
/********************************************/
/** fc_strncmp                             **/
/**                                        **/
/** Compare string 1 to string 2.          **/
/** Compare terminates on count N          **/
/**                                        **/
/** Return value:                          **/
/**   Less than 0    = str1 < str2         **/
/**   Zero           = str1 egual str2     **/
/**   Greater than 0 = str1 > str2         **/
/********************************************/
_forward_ int
fc_strncmp( char *str1,
            char *str2,
            int   cnt)
{
   int c1, c2;
   int dif;

   while( cnt--) {
      c1 = (int)*str1++ & 0xff;  
      c2 = (int)*str2++ & 0xff;
      if( (c1 | c2) == 0)
         return(0);  /* strings equal */
      if( (dif = c1 - c2) == 0)
         continue;   /* chars are equal */
      if( c1 == 0)
         return(-1); /* str1 < str2 */
      if( c2 == 0)
         return(1);  /* str1 > str2 */
      return(dif);
   }  
   return(0);        /* strings equal */
} /* fc_strncmp */

/********************************************/
/*  fc_strcpy                               */
/*                                          */
/*  Copy str2 to str1.     .                */
/*  Str2 must be a pointer to a null        */
/*  terminated string. It is the caller's   */
/*  responsibility to insure that str1 is   */
/*  large enough to hold str2.              */
/*                                          */
/*  Return value:                           */
/*     pointer to str1                      */
/********************************************/
_static_ char *
fc_strcpy( char *str1,    /* dest */
           char *str2)    /* src */
{
    char *temp;
    temp = str1;

    while( (*str1++ = *str2++) != '\0') {
      continue;
   }
   return(temp);
} /* fc_strcpy */

/************************************************/
/**  fc_setup_ring                             **/
/**                                            **/
/**  After ring has been configured, this      **/
/**  routine is called to initialize the ring  **/
/************************************************/
_static_ void
fc_setup_ring(
fc_dev_ctl_t *p_dev_ctl,    /* point to dev_ctl area */
int ring)
{
   FC_BRD_INFO * binfo;
   iCfgParam   * clp;
   RING  * rp;

   binfo = &BINFO;
   rp = &binfo->fc_ring[ring];
   clp = DD_CTL.p_config[binfo->fc_brd_no];

   /* set up the watchdog timer control structure section */
   if (!rp->fc_wdt_inited) {
      if (ring == FC_FCP_RING) {
         if(clp[CFG_LINKDOWN_TMO].a_current) {
            rp->fc_ringtmo = clp[CFG_LINKDOWN_TMO].a_current;
         }
         else {
            rp->fc_ringtmo = clp[CFG_LINKDOWN_TMO].a_default;
         }
      } else {
         rp->fc_ringtmo = RING_TMO_DFT;
      }
      RINGTMO = 0;
      rp->fc_wdt_inited = 1;
   }
}       /* End fc_setup_ring */

/************************************************************************/
/*                                                                      */
/* NAME:        fc_closewait                                            */
/*                                                                      */
/* FUNCTION:    Adapter Driver Wait for Close Routine                   */
/*      This routine waits for the adapter to finish all requests       */
/*                                                                      */
/* EXECUTION ENVIRONMENT:                                               */
/*      This routine can be called by a process.                        */
/*                                                                      */
/* INPUTS:                                                              */
/*      fc_dev_ctl_t - adapter unique data structure (one per adapter)  */
/*                                                                      */
/* RETURN VALUE DESCRIPTION:  none                                      */
/*                                                                      */
/* EXTERNAL PROCEDURES CALLED:                                          */
/*      disable_lock    lock_enable                                     */
/*                                                                      */
/************************************************************************/
_static_ void
fc_closewait(
fc_dev_ctl_t *p_dev_ctl)    /* point to dev_ctl area */
{
   FC_BRD_INFO * binfo;
   int  ipri;
   struct buf *bp, *nextbp;

   binfo = &BINFO;

   ipri = disable_lock(FC_LVL, &CMD_LOCK);

   /* wait for all operations to complete */
   while ((fc_ring_txcnt(binfo, FC_FCP_RING)
        || fc_ring_txpcnt(binfo, FC_FCP_RING)
        || binfo->fc_mbox.q_cnt)) {
      unlock_enable(ipri, &CMD_LOCK);
      DELAYMSctx(1000);            /* delay 1 second */
      ipri = disable_lock(FC_LVL, &CMD_LOCK);
   }

   /* Clear out timeout queue */
   for (bp = p_dev_ctl->timeout_head; bp != NULL; ) {
      nextbp = bp->av_forw;
      bp->b_error = ETIMEDOUT;
      bp->b_flags |= B_ERROR;
      bp->b_resid = bp->b_bcount;
      FCSTATCTR.fcpScsiTmo++;
      fc_do_iodone(bp);
      bp = nextbp;
   }
   p_dev_ctl->timeout_head = NULL;
   p_dev_ctl->timeout_count = 0;

   /* update the device state */
   binfo->fc_open_count &= ~FC_FCP_OPEN;
   if (binfo->fc_open_count == 0)
      p_dev_ctl->device_state = CLOSED;

   unlock_enable(ipri, &CMD_LOCK);
   return;

}  /* End fc_closewait */


/*
 * This routine copies data from src 
 * then potentially swaps the destination to big endian.
 * Assumes cnt is a multiple of sizeof(uint32).
 */
_static_ void
fc_pcimem_bcopy(
uint32  *src,
uint32  *dest,
uint32 cnt)
{
   uint32 ldata;
   int  i;

   for (i = 0; i < (int)cnt; i += sizeof(uint32)) {
      ldata = *src++;
      ldata = PCIMEM_LONG(ldata);
      *dest++ = ldata;
   }
}       /* End fc_pcimem_bcopy */


#define SCSI3_PERSISTENT_RESERVE_IN 0x5e

/******************************************************/
/**  handle_fcp_event                                **/
/**                                                  **/
/**  Description: Process an FCP Rsp Ring completion **/
/**                                                  **/
/******************************************************/
_static_ void
handle_fcp_event(
fc_dev_ctl_t *p_dev_ctl,
RING         *rp,
IOCBQ        *temp)
{
   FC_BRD_INFO   * binfo;
   iCfgParam     * clp;
   IOCB          * cmd;
   fc_buf_t      * fcptr;
   struct buf    * bp;
   T_SCSIBUF     * sbp;
   FCP_RSP       * fcpRsp;
   uint32        * lp, i, qfull;
   dvi_t         * dev_ptr, * next_dev_ptr;
   NODELIST      * ndlp;

   /* called from host_interrupt() to process R2ATT */
   binfo = &BINFO;
   cmd = &temp->iocb;
   qfull = 0;
   ndlp = 0;

   /* look up FCP complete by IoTag */
   if ((fcptr = fc_deq_fcbuf_active(rp, cmd->ulpIoTag)) == NULL) {
      /* ERROR: completion with missing FCP command */
      if (!((cmd->ulpStatus == IOSTAT_LOCAL_REJECT) && 
          ((cmd->un.grsp.perr.statLocalError == IOERR_INVALID_RPI) || 
          (cmd->un.grsp.perr.statLocalError == IOERR_ABORT_IN_PROGRESS) || 
          (cmd->un.grsp.perr.statLocalError == IOERR_SEQUENCE_TIMEOUT) || 
          (cmd->un.grsp.perr.statLocalError == IOERR_ABORT_REQUESTED)))) {
         /* Stray FCP completion */
         fc_log_printf_msg_vargs( binfo->fc_brd_no,
                &fc_msgBlk0720,                   /* ptr to msg structure */
                 fc_mes0720,                      /* ptr to msg */
                  fc_msgBlk0720.msgPreambleStr,   /* begin varargs */
                   cmd->ulpCommand,
                    cmd->ulpIoTag,
                     cmd->ulpStatus,
                      cmd->un.ulpWord[4]);        /* end varargs */
      }

      FCSTATCTR.fcpStrayCmpl++;
      return;
   }
   FCSTATCTR.fcpCmpl++;

   dev_ptr = fcptr->dev_ptr;
   dev_ptr->stop_send_io = 0;


   if(dev_ptr->queue_state == ACTIVE_PASSTHRU) {
      node_t          * map_node_ptr;
      struct dev_info * map_dev_ptr;

      map_node_ptr = (node_t *)dev_ptr->pend_head;
      map_dev_ptr  = (struct dev_info *)dev_ptr->pend_tail;
      dev_ptr->pend_head = 0;
      dev_ptr->pend_tail = 0;
      dev_ptr->queue_state = HALTED;
      dev_ptr->active_io_count--;
      if(map_dev_ptr)
         map_dev_ptr->active_io_count--;
      if(map_node_ptr)
         map_node_ptr->num_active_io--;

      dev_ptr->ioctl_event = cmd->ulpStatus;
      dev_ptr->ioctl_errno = (uint32)cmd->un.grsp.perr.statLocalError;
      fcpRsp = &fcptr->fcp_rsp;
      dev_ptr->sense_length = SWAP_DATA(fcpRsp->rspSnsLen);
      if(cmd->ulpCommand == CMD_FCP_IWRITE64_CX) {
         if (cmd->ulpStatus == IOSTAT_SUCCESS) {
            dev_ptr->clear_count  = 1;
         }
         else {
            dev_ptr->clear_count  = 0;
         }
      }
      else {
         dev_ptr->clear_count  = cmd->un.fcpi.fcpi_parm;
      }
      return;
   }

   /*
    * Is it a SCSI Report Lun command ?
    */
   if ((fcptr->fcp_cmd.fcpCdb[0] == FCP_SCSI_REPORT_LUNS) &&
       (fcptr->flags & FCBUF_INTERNAL)) {
     uchar     *datap;
     MBUF_INFO *mbufp;
     node_t    *nodep;
     uint32     rptLunLen;
     uint32    *datap32;
     uint32     max, lun;

     mbufp = (MBUF_INFO *)fcptr->sc_bufp;
     fcptr->sc_bufp = 0;
     mbufp->size = 4096;
     nodep = dev_ptr->nodep;
     if(nodep == 0) {
        fc_mem_put(binfo, MEM_IOCB, (uchar * )mbufp);
        dev_ptr->active_io_count--;
        fc_enq_fcbuf(fcptr);
        return;
     }
     if ((cmd->ulpStatus == IOSTAT_SUCCESS) || 
         ((cmd->ulpStatus == IOSTAT_FCP_RSP_ERROR) &&
          (fcptr->fcp_rsp.rspStatus2 & RESID_UNDER) &&
          (fcptr->fcp_rsp.rspStatus3 == SCSI_STAT_GOOD))) {

       datap = (uchar *)mbufp->virt;
       /*
        * if Lun0 uses VSA, we assume others use too.
        */
       if ((datap[8] & 0xc0) == 0x40) {
         nodep->addr_mode = VOLUME_SET_ADDRESSING; 
       }
       /*
        * Skip retry
        */
       datap32 = (uint32 *)mbufp->virt;
       rptLunLen = SWAP_DATA(*datap32);
       /* search for the max lun */
       max = 0;
       for(i=0; i < rptLunLen; i+=8) {
          datap32 += 2;
          lun = (((* datap32) >> FC_LUN_SHIFT) & 0xff);
	  if(lun > max)
             max = lun;
       }
       if(i) {
          nodep->max_lun = max + 1;
       }

       if(nodep->virtRptLunData == 0) {
	  if(rptLunLen > 8) {      /* more than 1 lun */
             nodep->virtRptLunData = mbufp->virt;
             nodep->physRptLunData = mbufp->phys;
	  } else {
             fc_free(p_dev_ctl, mbufp);
          }
       }
     } else {
        datap = 0;
        fc_free(p_dev_ctl, mbufp);
        nodep->virtRptLunData = 0;
        nodep->physRptLunData = 0;
     }

     fc_mem_put(binfo, MEM_IOCB, (uchar * )mbufp);

     dev_ptr->active_io_count--;
     nodep->num_active_io--;
     fc_enq_fcbuf(fcptr);
     
     if ((datap == 0) && (!(nodep->flags & RETRY_RPTLUN))) {
          nodep->flags |= RETRY_RPTLUN;
          /* Wait a little bit for ABTSs to settle down */
          fc_clk_set(p_dev_ctl, 1, issue_report_lun, (void *)dev_ptr, 0);
     }
     else {
        nodep->flags &= ~RETRY_RPTLUN;
        nodep->rptlunstate = REPORT_LUN_COMPLETE;
     }
     return; 
   }


   sbp = fcptr->sc_bufp;
   bp = (struct buf *) sbp;


   if (cmd->ulpStatus) {
      fcpRsp = &fcptr->fcp_rsp;
      {
         uint32 did;
         uint32 pan;
         uint32 sid;

         if ((dev_ptr->nodep) && (dev_ptr->nodep->nlp)) {
            ndlp = dev_ptr->nodep->nlp;
            did = ndlp->nlp_DID;
            pan = ndlp->id.nlp_pan;
            sid = ndlp->id.nlp_sid;
            if(ndlp->nlp_action & NLP_DO_RSCN)
               qfull = 1;
         } else {
            did = 0;
            pan = 0;
            sid = 0;
         }
         /* FCP cmd <Cbd0> failed on device (<sid>, <lun_id>) DID <did> */
         fc_log_printf_msg_vargs( binfo->fc_brd_no,
                &fc_msgBlk0729,                        /* ptr to msg structure */
                 fc_mes0729,                           /* ptr to msg */
                  fc_msgBlk0729.msgPreambleStr,        /* begin varargs */
                   fcptr->fcp_cmd.fcpCdb[0],
                    FC_SCSID(pan, sid),
                     (uint32)(dev_ptr->lun_id),
                      did,
                       (uint32)fcpRsp->rspInfo3,
                        (uint32)cmd->un.grsp.perr.statLocalError,
                         *((uint32 *)(((uint32 *)cmd) + WD6)),   /* IOCB wd 6 */
                          *((uint32 *)(((uint32 *)cmd) + WD7))); /* IOCB wd 7, end varargs */
      
      }
      lp = (uint32 * )fcpRsp;
      i = 0;
      /* FCP command failed: RSP */
      fc_log_printf_msg_vargs( binfo->fc_brd_no,
             &fc_msgBlk0730,                    /* ptr to msg structure */
              fc_mes0730,                       /* ptr to msg */
               fc_msgBlk0730.msgPreambleStr,    /* begin varargs */
                lp[2],
                 lp[3],
                  lp[4],
                   lp[5]);                      /* end varargs */
      if (fcpRsp->rspStatus2 & RSP_LEN_VALID) {
         i = SWAP_DATA(fcpRsp->rspRspLen);
      }
      if (fcpRsp->rspStatus2 & SNS_LEN_VALID) {
         lp = (uint32 * )(((uchar * ) & fcpRsp->rspInfo0) + i);
         /* FCP command failed: SNS */
         fc_log_printf_msg_vargs( binfo->fc_brd_no,
                &fc_msgBlk0731,                    /* ptr to msg structure */
                 fc_mes0731,                       /* ptr to msg */
                  fc_msgBlk0731.msgPreambleStr,    /* begin varargs */
                   lp[0],
                    lp[1],
                     lp[2],
                      lp[3],
                       lp[4],
                        lp[5],
                         lp[6],
                          lp[7]);                  /* end varargs */
         if (i > sizeof(FCP_RSP)) {
            cmd->ulpStatus = IOSTAT_REMOTE_STOP;
            goto handle_iocb;
         }

         if(binfo->fc_process_LA == 0)
            goto skip4;  /* link down processing */
         if (dev_ptr->first_check & FIRST_CHECK_COND) {
            clp = DD_CTL.p_config[binfo->fc_brd_no];
            dev_ptr->first_check &= ~FIRST_CHECK_COND;
            if((clp[CFG_FIRST_CHECK].a_current) &&
               (SWAP_DATA((lp[3]) & SWAP_DATA(0xFF000000)) == 0x29000000)) {
               FCSTATCTR.fcpFirstCheck++;

               lp = (uint32 *)&cmd->un.ulpWord[4];
               /* Retry FCP command due to 29,00 check condition */
               fc_log_printf_msg_vargs( binfo->fc_brd_no,
                      &fc_msgBlk0732,                  /* ptr to msg structure */
                       fc_mes0732,                     /* ptr to msg */
                        fc_msgBlk0732.msgPreambleStr,  /* begin varargs */
                         *lp,
                          *(lp+1),
                           *(lp+2),
                            *(lp+3));                  /* end varargs */
               fc_fcp_bufunmap(p_dev_ctl, sbp);


               /*
                * Queue this command to the head of the device's 
                * pending queue for processing by fc_issue_cmd.
                */
               if (dev_ptr->pend_head == NULL) { /* Is queue empty? */
                  dev_ptr->pend_head = sbp;
                  dev_ptr->pend_tail = sbp;
                  bp->av_forw = NULL;
                  fc_enq_wait(dev_ptr);
               } else {                    /* Queue not empty */
                  bp->av_forw = (struct buf *) dev_ptr->pend_head;
                  dev_ptr->pend_head = sbp;
               }
               dev_ptr->pend_count++;
               dev_ptr->active_io_count--;
               dev_ptr->nodep->num_active_io--;
               fc_enq_fcbuf(fcptr);
               return;
            }
         }

         fc_bcopy(((uchar * ) & fcpRsp->rspInfo0) + i, dev_ptr->sense,
             MAX_FCP_SNS);
         dev_ptr->sense_valid = 1;
         dev_ptr->sense_length = SWAP_DATA(fcpRsp->rspSnsLen);

      } else {
         if ((cmd->ulpStatus == IOSTAT_FCP_RSP_ERROR) &&
            ((((uchar)fcpRsp->rspStatus3) & SCSI_STATUS_MASK) == SCSI_STAT_QUE_FULL) &&
            (dev_ptr->qfull_retries > 0) &&
            (sbp->qfull_retry_count < dev_ptr->qfull_retries)) {
            clp = DD_CTL.p_config[binfo->fc_brd_no];
            if (clp[CFG_DQFULL_THROTTLE].a_current) {
               if (dev_ptr->fcp_cur_queue_depth > FC_MIN_QFULL) {
                  if(dev_ptr->active_io_count > FC_MIN_QFULL)
                     dev_ptr->fcp_cur_queue_depth = dev_ptr->active_io_count - 1;
                  else
                     dev_ptr->fcp_cur_queue_depth = FC_MIN_QFULL;
               }
            }

            fc_qfull_retry((void *)fcptr);

            sbp->qfull_retry_count++;

            dev_ptr->qfullcnt++;
            dev_ptr->active_io_count--;
            dev_ptr->nodep->num_active_io--;
            fc_enq_fcbuf(fcptr);
            return;
         }
      }
   } else {
      fcpRsp = &fcptr->fcp_rsp;
   }

handle_iocb:

   switch (cmd->ulpStatus) {
   case IOSTAT_SUCCESS:
      FCSTATCTR.fcpGood++;
      break;

   case IOSTAT_FCP_RSP_ERROR:
      /* ERROR: all is not well with the FCP Response */
      bp->b_error = EIO;
      bp->b_flags |= B_ERROR;

      bp->b_resid = 0;
      clp = DD_CTL.p_config[binfo->fc_brd_no];
      FCSTATCTR.fcpRspErr++;

      if(binfo->fc_process_LA == 0)
         goto skip4;  /* link down processing */


      if ((fcpRsp->rspStatus2 & RESID_UNDER) ||
          (fcpRsp->rspStatus2 & RESID_OVER)) {
         if (fcpRsp->rspStatus2 & RESID_UNDER) {
             /* This is not an error! */
             bp->b_resid = SWAP_DATA(fcpRsp->rspResId);

             bp->b_error = 0;
             bp->b_flags &= ~B_ERROR;
             /* FCP Read Underrun */
             fc_log_printf_msg_vargs( binfo->fc_brd_no,
                    &fc_msgBlk0733,                      /* ptr to msg structure */
                     fc_mes0733,                         /* ptr to msg */
                      fc_msgBlk0733.msgPreambleStr,      /* begin varargs */
                       *((uint32 *)(((uint32 *)cmd) + WD7)), /* IOCB wd 7 */
                        (uint32)cmd->ulpContext,
                         SWAP_DATA(fcpRsp->rspResId),
                          cmd->un.fcpi.fcpi_parm);       /* end varargs */
         }
         /* Overrun already has error set */
      }
      else {
         if ((bp->b_flags & B_READ) && cmd->un.fcpi.fcpi_parm) {
            /* This is ALWAYS a readcheck error!! */

            /* Give Check Condition priority over Read Check */
            if (fcpRsp->rspStatus3 != SCSI_STAT_CHECK_COND) {
               bp->b_error = EIO;
               bp->b_flags |= B_ERROR;
               bp->b_resid = bp->b_bcount;
               /* FCP Read Check Error */
               fc_log_printf_msg_vargs( binfo->fc_brd_no,
                      &fc_msgBlk0734,                      /* ptr to msg structure */
                       fc_mes0734,                         /* ptr to msg */
                        fc_msgBlk0734.msgPreambleStr,      /* begin varargs */
                         *((uint32 *)(((uint32 *)cmd) + WD7)), /* IOCB wd 7 */
                          (uint32)cmd->ulpContext,
                           SWAP_DATA(fcpRsp->rspResId), 
                            cmd->un.fcpi.fcpi_parm);       /* end varargs */
            }
            else {
               /* FCP Read Check Error with Check Condition */
               fc_log_printf_msg_vargs( binfo->fc_brd_no,
                      &fc_msgBlk0735,                      /* ptr to msg structure */
                       fc_mes0735,                         /* ptr to msg */
                        fc_msgBlk0735.msgPreambleStr,      /* begin varargs */
                         *((uint32 *)(((uint32 *)cmd) + WD7)), /* IOCB wd 7 */
                          (uint32)cmd->ulpContext,
                           SWAP_DATA(fcpRsp->rspResId), 
                            cmd->un.fcpi.fcpi_parm);       /* end varargs */
            }
         }
      }

      /* For QUE_FULL we will delay the iodone */
      if((((uchar) fcpRsp->rspStatus3) & SCSI_STATUS_MASK) == SCSI_STAT_QUE_FULL) {
         dev_ptr->qfullcnt++;
         if (clp[CFG_DQFULL_THROTTLE].a_current) {
            if (dev_ptr->fcp_cur_queue_depth > FC_MIN_QFULL) {
               if(dev_ptr->active_io_count > 1)
                  dev_ptr->fcp_cur_queue_depth = dev_ptr->active_io_count - 1;
               else
                  dev_ptr->fcp_cur_queue_depth = 1;
            }
            if (dev_ptr->active_io_count > FC_MIN_QFULL) {
               /*
                * Try out
                * stop_send_io will be decreament by 1 in fc_q_depth_up();
                */
               dev_ptr->stop_send_io = clp[CFG_NO_DEVICE_DELAY].a_current;
            }
         }
         /* FCP QUEUE Full */
         fc_log_printf_msg_vargs( binfo->fc_brd_no,
                &fc_msgBlk0736,                             /* ptr to msg structure */
                 fc_mes0736,                                /* ptr to msg */
                  fc_msgBlk0736.msgPreambleStr,             /* begin varargs */
                   dev_ptr->fcp_cur_queue_depth,
                    dev_ptr->active_io_count,
                     dev_ptr->flags,
                      clp[CFG_DQFULL_THROTTLE].a_current);  /* end varargs */
         qfull = 1;
         /* Set any necessary flags for buf error */
         bp->b_error = EBUSY;
         bp->b_flags |= B_ERROR;
         bp->b_resid = bp->b_bcount;
      }
      lpfc_handle_fcp_error(pkt, fcptr, cmd);

      if ((fcpRsp->rspStatus2 & RSP_LEN_VALID) && 
          (fcpRsp->rspInfo3 != RSP_NO_FAILURE)) {

         /* Error detected in the FCP layer */
         sbp->status_validity = SC_ADAPTER_ERROR;

         if(clp[CFG_DELAY_RSP_ERR].a_current)
            qfull = clp[CFG_DELAY_RSP_ERR].a_current;

         switch (fcpRsp->rspInfo3) {
         case RSP_TM_NOT_SUPPORTED:
        SET_ADAPTER_STATUS(sbp, SC_NO_DEVICE_RESPONSE)
            break;

         case RSP_DATA_BURST_ERR:
         case RSP_CMD_FIELD_ERR:
         case RSP_RO_MISMATCH_ERR:
            if (fcpRsp->rspStatus3 != SCSI_STAT_GOOD) {
               sbp->status_validity = SC_SCSI_ERROR;
               sbp->scsi_status = fcpRsp->rspStatus3;
               if ((fcpRsp->rspStatus3 == SC_CHECK_CONDITION) || 
                   (fcpRsp->rspStatus3 == SC_COMMAND_TERMINATED)) {
                  sbp->adap_q_status = SC_DID_NOT_CLEAR_Q;
               }


               break;
            }

         case RSP_TM_NOT_COMPLETED:
         default:
        SET_ADAPTER_STATUS(sbp, SC_ADAPTER_HDW_FAILURE)
            break;
         }
      } else if (fcpRsp->rspStatus3 != SCSI_STAT_GOOD) {
         /* SCSI layer detected error */
         if (fcpRsp->rspStatus3 == SCSI_STAT_CHECK_COND) {
            uint32 cc;
            /* FCP error: Check condition */
            fc_log_printf_msg_vargs( binfo->fc_brd_no,
                   &fc_msgBlk0737,                                   /* ptr to msg structure */
                    fc_mes0737,                                      /* ptr to msg */
                     fc_msgBlk0737.msgPreambleStr,                   /* begin varargs */
                      *((uint32 *)(((uint32 *)cmd) + WD7)),          /* IOCB wd 7 */
                       (uint32)cmd->ulpIoTag,
                        (uint32)cmd->ulpContext,
                         (uint32)cmd->un.grsp.perr.statLocalError);  /* end varargs */
            i = SWAP_DATA(fcpRsp->rspRspLen);
            lp = (uint32 * )(((uchar * ) & fcpRsp->rspInfo0) + i);

            cc = (SWAP_DATA((lp[3]) & SWAP_DATA(0xFF000000)));
            switch(cc) {
            case 0x29000000: /* Power on / reset */
               i = 0;
               /* 29,00 Check condition received */
               fc_log_printf_msg_vargs( binfo->fc_brd_no,
                      &fc_msgBlk0738,                  /* ptr to msg structure */
                       fc_mes0738,                     /* ptr to msg */
                        fc_msgBlk0738.msgPreambleStr,  /* begin varargs */
                         lp[0],
                          lp[1],
                           lp[2],
                            lp[3]);                    /* end varargs */
               break;
            case 0x0:        /* No additional sense info */
               if((lp[3]) & SWAP_DATA(0x00FF0000))  /* if ASCQ != 0 */
                  goto default_chk;
            case 0x44000000: /* Internal Target failure */
            case 0x25000000: /* Login Unit Not Support */
            case 0x20000000: /* Invalid Command operation code */
               if ((cc == 0x20000000) && (fcptr->fcp_cmd.fcpCdb[0] == SCSI3_PERSISTENT_RESERVE_IN)) {
                  /* Check condition received ERR1 */
                  fc_log_printf_msg_vargs( binfo->fc_brd_no,
                         &fc_msgBlk0739,                 /* ptr to msg structure */
                          fc_mes0739,                    /* ptr to msg */
                           fc_msgBlk0739.msgPreambleStr, /* begin varargs */
                            lp[0],
                             lp[1],
                              lp[2],
                               lp[3]);                   /* end varargs */
                  goto out;
               }
               if(clp[CFG_CHK_COND_ERR].a_current) {
                  /* We want to return check cond on TUR cmd */
                  if (fcptr->fcp_cmd.fcpCdb[0] == FCP_SCSI_TEST_UNIT_READY)
                     goto default_chk; 
                  fc_bzero((void * )dev_ptr->sense, MAX_FCP_SNS);
                  dev_ptr->sense_valid = 0;
                  dev_ptr->sense_length = 0;
                  fcpRsp->rspStatus3 = SC_COMMAND_TERMINATED;
                  bp->b_error = EIO;
                  bp->b_flags |= B_ERROR;
                  bp->b_resid = bp->b_bcount;
                  sbp->status_validity = SC_ADAPTER_ERROR;
              SET_ADAPTER_STATUS(sbp, SC_SCSI_BUS_RESET)
                  i = 0;
                  if(clp[CFG_DELAY_RSP_ERR].a_current)
                     qfull = clp[CFG_DELAY_RSP_ERR].a_current;
                  /* Check condition received ERR2 */
                  fc_log_printf_msg_vargs( binfo->fc_brd_no,
                         &fc_msgBlk0740,                 /* ptr to msg structure */
                          fc_mes0740,                    /* ptr to msg */
                           fc_msgBlk0740.msgPreambleStr, /* begin varargs */
                            lp[0],
                             lp[1],
                              lp[2],
                               lp[3]);                   /* end varargs */
                  goto out;
               }
            default:
               if(clp[CFG_DELAY_RSP_ERR].a_current)
                  qfull = clp[CFG_DELAY_RSP_ERR].a_current;
default_chk:
               i = 0;
               /* Check condition received */
               fc_log_printf_msg_vargs( binfo->fc_brd_no,
                      &fc_msgBlk0741,                    /* ptr to msg structure */
                       fc_mes0741,                       /* ptr to msg */
                        fc_msgBlk0741.msgPreambleStr,    /* begin varargs */
                         lp[0],
                          lp[1],
                           lp[2],
                            lp[3]);                      /* end varargs */
               break;
            }
         }
         else {
            if(clp[CFG_DELAY_RSP_ERR].a_current)
               qfull = clp[CFG_DELAY_RSP_ERR].a_current;
         }

         sbp->status_validity = SC_SCSI_ERROR;
         sbp->scsi_status = fcpRsp->rspStatus3;
         if ((fcpRsp->rspStatus3 == SC_CHECK_CONDITION) || 
             (fcpRsp->rspStatus3 == SC_COMMAND_TERMINATED)) {
            sbp->adap_q_status = SC_DID_NOT_CLEAR_Q;

         }
      }
      break;

   case IOSTAT_REMOTE_STOP:
      /* ERROR: ABTS/ABTX by remote N_PORT */
      FCSTATCTR.fcpRemoteStop++;
      bp->b_error = EIO;
      bp->b_flags |= B_ERROR;
      bp->b_resid = bp->b_bcount;
      sbp->status_validity = SC_SCSI_ERROR;
      sbp->scsi_status = SC_COMMAND_TERMINATED;
      sbp->adap_q_status = SC_DID_NOT_CLEAR_Q;

      break;

   case IOSTAT_LOCAL_REJECT:
      FCSTATCTR.fcpLocalErr++;
      switch (cmd->un.grsp.perr.statLocalError) {
      case IOERR_SEQUENCE_TIMEOUT:
         FCSTATCTR.fcpLocalTmo++;
         /* E_D_TOV timeout */
         bp->b_error = ETIMEDOUT;
         sbp->adap_q_status = SC_DID_NOT_CLEAR_Q; 
         bp->b_flags |= B_ERROR;
         bp->b_resid = bp->b_bcount;
         sbp->status_validity = SC_ADAPTER_ERROR;
     SET_ADAPTER_STATUS(sbp, SC_CMD_TIMEOUT)
         break;

      case IOERR_NO_RESOURCES:
         FCSTATCTR.fcpLocalNores++;
         fc_qfull_retry((void *)fcptr);
         dev_ptr->active_io_count--;
         dev_ptr->nodep->num_active_io--;
         fc_enq_fcbuf(fcptr);
         return;
      case IOERR_BUFFER_SHORTAGE:
         FCSTATCTR.fcpLocalBufShort++;
         /* The adapter is too busy to deal with this command */
         bp->b_error = EBUSY;
         bp->b_flags |= B_ERROR;
         bp->b_resid = bp->b_bcount;
         sbp->status_validity = 0;
         break;

      case IOERR_MISSING_CONTINUE:
      case IOERR_ILLEGAL_COMMAND:
      case IOERR_ILLEGAL_FIELD:
      case IOERR_BAD_CONTINUE:
      case IOERR_TOO_MANY_BUFFERS:
      case IOERR_EXTRA_DATA:
      case IOERR_ILLEGAL_LENGTH:
      case IOERR_UNSUPPORTED_FEATURE:
         /* Let's call these driver software errors */
         qfull = 1;
         FCSTATCTR.fcpLocalSfw++;
         bp->b_error = EINVAL;
         bp->b_flags |= B_ERROR;
         bp->b_resid = bp->b_bcount;
         sbp->status_validity = 0;

         {
            uint32 did;

            did = 0;
            if ((dev_ptr->nodep) && (dev_ptr->nodep->nlp))
               did = dev_ptr->nodep->nlp->nlp_DID;
            /* FCP completion error */
            fc_log_printf_msg_vargs( binfo->fc_brd_no,
                   &fc_msgBlk0742,                    /* ptr to msg structure */
                    fc_mes0742,                       /* ptr to msg */
                     fc_msgBlk0742.msgPreambleStr,    /* begin varargs */
                      cmd->ulpStatus,
                       cmd->un.ulpWord[4],
                        did );                        /* end varargs */
         }
         break;

      case IOERR_TX_DMA_FAILED:
         FCSTATCTR.fcpLocalTxDMA++;
         goto skip2;
      case IOERR_RX_DMA_FAILED:
         FCSTATCTR.fcpLocalRxDMA++;
         goto skip2;
      case IOERR_INTERNAL_ERROR:
         FCSTATCTR.fcpLocalinternal++;
         goto skip2;
      case IOERR_CORRUPTED_DATA:
      case IOERR_CORRUPTED_RPI:
         FCSTATCTR.fcpLocalCorrupt++;
skip2:
         /* Let's call these adapter hardware errors */
         bp->b_error = EIO;
         bp->b_flags |= B_ERROR;
         bp->b_resid = bp->b_bcount;
         sbp->status_validity = SC_ADAPTER_ERROR;
     SET_ADAPTER_STATUS(sbp, SC_ADAPTER_HDW_FAILURE)

         {
            uint32 did;

            did = 0;
            if ((dev_ptr->nodep) && (dev_ptr->nodep->nlp))
               did = dev_ptr->nodep->nlp->nlp_DID;
            /* FCP completion error */
            fc_log_printf_msg_vargs( binfo->fc_brd_no,
                   &fc_msgBlk0743,                    /* ptr to msg structure */
                    fc_mes0743,                       /* ptr to msg */
                     fc_msgBlk0743.msgPreambleStr,    /* begin varargs */
                      cmd->ulpStatus,
                       cmd->un.ulpWord[4],
                        did );                        /* end varargs */
         }

         break;

      case IOERR_ILLEGAL_FRAME:
         FCSTATCTR.fcpLocalIllFrm++;
         goto skip3;
      case IOERR_DUP_FRAME:
         FCSTATCTR.fcpLocalDupFrm++;
         goto skip3;
      case IOERR_LINK_CONTROL_FRAME:
         FCSTATCTR.fcpLocalLnkCtlFrm++;
skip3:
         qfull = 1;
         /* Let's call these device hardware errors */
         bp->b_error = EINVAL;
         bp->b_flags |= B_ERROR;
         bp->b_resid = bp->b_bcount;
         sbp->status_validity = 0;

         {
            uint32 did;

            did = 0;
            if ((dev_ptr->nodep) && (dev_ptr->nodep->nlp))
               did = dev_ptr->nodep->nlp->nlp_DID;

            lp = (uint32 *)&cmd->un.ulpWord[4];
            /* FCP completion error */
            fc_log_printf_msg_vargs( binfo->fc_brd_no,
                   &fc_msgBlk0744,                   /* ptr to msg structure */
                    fc_mes0744,                      /* ptr to msg */
                     fc_msgBlk0744.msgPreambleStr,   /* begin varargs */
                      did,
                       *lp,
                        *(lp+2),
                         *(lp+3) );                  /* end varargs */
         }

         break;

      case IOERR_LOOP_OPEN_FAILURE:
         FCSTATCTR.fcpLocalLoopOpen++;
         /* The device disappeared from the loop! */
         qfull = 1;
         bp->b_error = EIO;
         bp->b_flags |= B_ERROR;
         bp->b_resid = bp->b_bcount;
         sbp->status_validity = SC_ADAPTER_ERROR;
     SET_ADAPTER_STATUS(sbp, SC_NO_DEVICE_RESPONSE)
         if(dev_ptr->nodep && (dev_ptr->nodep->flags & FC_NODEV_TMO)) {
            break;
         }
         if(binfo->fc_ffstate != FC_READY) {
            break;
         }
         /* Will HALT, CLEARQ, and kick off discovery, below */
         /* Try to relogin, and if unsuccessful reject future cmds */
         if((ndlp == 0) && dev_ptr->nodep)
            ndlp = fc_findnode_rpi(binfo, (uint32)dev_ptr->nodep->rpi);

         if ((ndlp) && !(ndlp->nlp_flag & (NLP_NODEV_TMO | NLP_REQ_SND))) {
            ndlp->nlp_flag &= ~NLP_RM_ENTRY;
            /* We are in FC_READY state */
            if (!(ndlp->nlp_action & NLP_DO_RSCN)) {
               binfo->fc_flag |= FC_RSCN_MODE;
               ndlp->nlp_action |= NLP_DO_RSCN;
               fc_nextrscn(p_dev_ctl, 1);
            }
         }
         break;

      case IOERR_INVALID_RPI:
         FCSTATCTR.fcpLocalInvalRpi++;
         goto skip4;
      case IOERR_LINK_DOWN:
         FCSTATCTR.fcpLocalLinkDown++;
skip4:
         /* Retry these failures */
         qfull=1;
         bp->b_error = EIO;
         bp->b_flags |= B_ERROR;
         bp->b_resid = bp->b_bcount;
         sbp->status_validity = SC_ADAPTER_ERROR;
     SET_ADAPTER_STATUS(sbp, SC_SCSI_BUS_RESET)
         break;

      case IOERR_OUT_OF_ORDER_DATA:
      case IOERR_OUT_OF_ORDER_ACK:
         FCSTATCTR.fcpLocalOOO++;
         /* Retry these failures */
         bp->b_error = ENXIO;
         bp->b_flags |= B_ERROR;
         bp->b_resid = bp->b_bcount;
         sbp->status_validity = 0;
         break;

      case IOERR_ABORT_IN_PROGRESS:
         FCSTATCTR.fcpLocalAbtInp++;
         goto skip5;
      case IOERR_ABORT_REQUESTED:
         FCSTATCTR.fcpLocalAbtReq++;
skip5:
         /* Abort requested by us */
         if (fcptr->flags & FCBUF_ABTS) {
            /* ABTS sent because of operation timeout */
            bp->b_error = ETIMEDOUT;
            sbp->status_validity = SC_ADAPTER_ERROR;
        SET_ADAPTER_STATUS(sbp, SC_CMD_TIMEOUT)
         } else {
            bp->b_error = ENXIO;
            sbp->status_validity = 0;
         }
         bp->b_flags |= B_ERROR;
         bp->b_resid = bp->b_bcount;
         break;

      case IOERR_SUCCESS:
      case IOERR_NO_XRI:
      case IOERR_XCHG_DROPPED:
      case IOERR_RCV_BUFFER_WAITING:
      case IOERR_RECEIVE_BUFFER_TIMEOUT:
      case IOERR_RING_RESET:
      case IOERR_BAD_HOST_ADDRESS:
      case IOERR_RCV_HDRBUF_WAITING:
      case IOERR_MISSING_HDR_BUFFER:
      case IOERR_MSEQ_CHAIN_CORRUPTED:
      case IOERR_ABORTMULT_REQUESTED:
      default:
         FCSTATCTR.fcpLocal++;
         bp->b_error = EIO;
         bp->b_flags |= B_ERROR;
         bp->b_resid = bp->b_bcount;
         sbp->status_validity = SC_ADAPTER_ERROR;
         SET_ADAPTER_STATUS(sbp, SC_NO_DEVICE_RESPONSE)

         {
            uint32 did;

            did = 0;
            if ((dev_ptr->nodep) && (dev_ptr->nodep->nlp))
               did = dev_ptr->nodep->nlp->nlp_DID;
            /* FCP completion error */
            fc_log_printf_msg_vargs( binfo->fc_brd_no,
                   &fc_msgBlk0745,                   /* ptr to msg structure */
                    fc_mes0745,                      /* ptr to msg */
                     fc_msgBlk0745.msgPreambleStr,   /* begin varargs */
                      cmd->ulpStatus,
                       cmd->un.ulpWord[4],
                        did );                       /* end varargs */
         }
         break;
      }
      break;

   case IOSTAT_NPORT_RJT:
   case IOSTAT_FABRIC_RJT:
      FCSTATCTR.fcpPortRjt++;
      /* The fabric or port rejected this command */
      if (cmd->un.grsp.perr.statAction == RJT_RETRYABLE) {
         bp->b_error = ENXIO;
         sbp->status_validity = SC_SCSI_ERROR;
         sbp->scsi_status = SC_BUSY_STATUS;
      } else {
         bp->b_error = EIO;
         sbp->status_validity = 0;
      }
      bp->b_flags |= B_ERROR;
      bp->b_resid = bp->b_bcount;
      break;

   case IOSTAT_NPORT_BSY:
   case IOSTAT_FABRIC_BSY:
      FCSTATCTR.fcpPortBusy++;
      /* The fabric or port is too busy to deal with this command */
      bp->b_error = ENXIO;
      bp->b_flags |= B_ERROR;
      bp->b_resid = bp->b_bcount;
      sbp->status_validity = SC_SCSI_ERROR;
      sbp->scsi_status = SC_BUSY_STATUS;
      break;

   case IOSTAT_INTERMED_RSP:
   case IOSTAT_LS_RJT:
   case IOSTAT_BA_RJT:
   default:
      FCSTATCTR.fcpError++;
      /* ERROR: None of these errors should occur! */
      bp->b_error = EIO;
      bp->b_flags |= B_ERROR;
      bp->b_resid = bp->b_bcount;
      sbp->status_validity = SC_ADAPTER_ERROR;
      SET_ADAPTER_STATUS(sbp, SC_NO_DEVICE_RESPONSE)

      {
         uint32 did;

         did = 0;
         if ((dev_ptr->nodep) && (dev_ptr->nodep->nlp))
            did = dev_ptr->nodep->nlp->nlp_DID;
         /* FCP completion error */
         fc_log_printf_msg_vargs( binfo->fc_brd_no,
                &fc_msgBlk0746,                   /* ptr to msg structure */
                 fc_mes0746,                      /* ptr to msg */
                  fc_msgBlk0746.msgPreambleStr,   /* begin varargs */
                   cmd->ulpStatus,
                    cmd->un.ulpWord[4],
                     did );                       /* end varargs */
      }
      break;
   }
out:

   if (fcptr->fcp_cmd.fcpCntl2)
   {
      /* This is a task management command */
      if (bp->b_flags & B_ERROR)
         dev_ptr->ioctl_errno = bp->b_error;
      else
         dev_ptr->ioctl_errno = 0;



      if (fcptr->fcp_cmd.fcpCntl2 & TARGET_RESET) {
         /* Cmpl Target Reset */
         fc_log_printf_msg_vargs( binfo->fc_brd_no,
                &fc_msgBlk0747,                       /* ptr to msg structure */
                 fc_mes0747,                          /* ptr to msg */
                  fc_msgBlk0747.msgPreambleStr,       /* begin varargs */
                   (uint32)dev_ptr->nodep->scsi_id,
                    (uint32)dev_ptr->lun_id,
                     (uint32)cmd->un.grsp.perr.statLocalError,
                      *((uint32 *)(((uint32 *)cmd) + WD7))); /* end varargs */
         clp = DD_CTL.p_config[binfo->fc_brd_no];
         dev_ptr->flags &= ~SCSI_TARGET_RESET;
         for (next_dev_ptr = dev_ptr->nodep->lunlist; next_dev_ptr != NULL; 
             next_dev_ptr = next_dev_ptr->next) {

            next_dev_ptr->flags &= ~SCSI_TARGET_RESET;
            /* First send ABTS on any outstanding I/O in txp queue */
            fc_abort_fcp_txpq(binfo, next_dev_ptr);
            fc_fail_cmd(next_dev_ptr, ENXIO, STAT_DEV_RESET);
            next_dev_ptr->fcp_cur_queue_depth =
               (ushort)clp[CFG_DFT_LUN_Q_DEPTH].a_current;
            if (next_dev_ptr->ioctl_wakeup == 0)
               fc_restart_device(next_dev_ptr);
         }
      }

      if (fcptr->fcp_cmd.fcpCntl2 & LUN_RESET) {
         /* Cmpl LUN Reset */
         fc_log_printf_msg_vargs( binfo->fc_brd_no,
                &fc_msgBlk0748,                       /* ptr to msg structure */
                 fc_mes0748,                          /* ptr to msg */
                  fc_msgBlk0748.msgPreambleStr,       /* begin varargs */
                   (uint32)dev_ptr->nodep->scsi_id,
                    (uint32)dev_ptr->lun_id,
                     (uint32)cmd->un.grsp.perr.statLocalError,
                      *((uint32 *)(((uint32 *)cmd) + WD7))); /* end varargs */
         dev_ptr->flags &= ~SCSI_LUN_RESET;
         /* First send ABTS on any outstanding I/O in txp queue */
         fc_abort_fcp_txpq(binfo, dev_ptr);
         fc_fail_cmd(dev_ptr, ENXIO, STAT_DEV_RESET);
         if (dev_ptr->ioctl_wakeup == 0)
            fc_restart_device(dev_ptr);
      }

      if (fcptr->fcp_cmd.fcpCntl2 & ABORT_TASK_SET) {
         /* Cmpl Abort Task Set */
         fc_log_printf_msg_vargs( binfo->fc_brd_no,
                &fc_msgBlk0749,                       /* ptr to msg structure */
                 fc_mes0749,                          /* ptr to msg */
                  fc_msgBlk0749.msgPreambleStr,       /* begin varargs */
                   (uint32)dev_ptr->nodep->scsi_id,
                    (uint32)dev_ptr->lun_id,
                     (uint32)cmd->un.grsp.perr.statLocalError,
                      *((uint32 *)(((uint32 *)cmd) + WD7))); /* end varargs */
         dev_ptr->flags &= ~SCSI_ABORT_TSET;
         /* First send ABTS on any outstanding I/O in txp queue */
         fc_abort_fcp_txpq(binfo, dev_ptr);
         fc_fail_cmd(dev_ptr, ENXIO, STAT_DEV_RESET);
         if (dev_ptr->ioctl_wakeup == 0)
            fc_restart_device(dev_ptr);
      }

      if (dev_ptr->ioctl_wakeup == 1) {
         dev_ptr->ioctl_wakeup = 0;
         fc_admin_wakeup(p_dev_ctl, dev_ptr, sbp);
      }
   } else {
      if ((bp->b_flags & B_ERROR) && 
          (dev_ptr->queue_state != STOPPING)) {
         /* An error has occurred, so halt the queues */
         sbp->adap_q_status = SC_DID_NOT_CLEAR_Q;
         if(qfull)
            fc_delay_iodone(p_dev_ctl, sbp);
         else
            fc_do_iodone(bp);
      } else {
         if(qfull)
            fc_delay_iodone(p_dev_ctl, sbp);
         else
            fc_do_iodone(bp);
      }
   }


   dev_ptr->active_io_count--;
   dev_ptr->nodep->num_active_io--;
   fc_enq_fcbuf(fcptr);


   if ((dev_ptr->nodep->tgt_queue_depth) &&
       (dev_ptr->nodep->tgt_queue_depth == dev_ptr->nodep->num_active_io)) {
      re_issue_fcp_cmd(dev_ptr->nodep->last_dev);
   }
   return;
}       /* End handle_fcp_event */


int
fc_delay_iodone(
fc_dev_ctl_t *p_dev_ctl,
T_SCSIBUF * sbp)
{
   FC_BRD_INFO   * binfo;
   iCfgParam     * clp;
   uint32          tmout;

   binfo = &BINFO;
   clp = DD_CTL.p_config[binfo->fc_brd_no];

   if(clp[CFG_NO_DEVICE_DELAY].a_current) {
      /* Need to set a timer so iodone can be called
       * for buffer upon expiration.
       */
     tmout = clp[CFG_NO_DEVICE_DELAY].a_current;

     if(fc_clk_set(p_dev_ctl, tmout,
        lpfc_scsi_selto_timeout, (void *)sbp, 0) != 0)
        return(1);
   }
   fc_do_iodone((struct buf *)sbp);
   return(0);
}       /* End fc_delay_iodone */


/**********************************************/
/**  handle_iprcv_seq                        **/
/**                                          **/
/**********************************************/
_static_ int
handle_iprcv_seq(
fc_dev_ctl_t *p_dev_ctl,
RING         *rp,
IOCBQ        *temp)
{
   MAILBOXQ      * mb;
   FC_BRD_INFO   * binfo;
   IOCB          * cmd = 0;
   IOCB          * savecmd;
   IOCBQ         * savetemp;
   NETHDR        * nh;
   fcipbuf_t     * p_mbuf, *mp, *last_mp;
   ndd_t         * p_ndd;
   NODELIST      * ndlp;
   MATCHMAP      * matp;
   uchar         * daddr;
   uchar         * saddr;
   int  count, i, la;

   binfo = &BINFO;
   p_ndd = (ndd_t * ) & (NDD);

   p_mbuf = 0;
   matp = (MATCHMAP *)0; /* prevent compiler warning */

   if (++NDDSTAT.ndd_recvintr_lsw == 0) {
      NDDSTAT.ndd_recvintr_msw++;
   }

   mp = 0;
   last_mp = 0;
   count = 0;
   la = 0;

   savetemp = temp;
   if (binfo->fc_ffstate < FC_READY) {
      if (binfo->fc_ffstate < rp->fc_xmitstate) {
         goto dropout;
      }
      la = 1;
   }

   savecmd = &temp->iocb;
   while (temp) {
      cmd = &temp->iocb;
      if (cmd->ulpStatus) {
         if ((cmd->ulpStatus == IOSTAT_LOCAL_REJECT) && 
             ((cmd->un.ulpWord[4] & 0xff) == IOERR_RCV_BUFFER_WAITING)) {
            FCSTATCTR.NoRcvBuf++;
            if(!(binfo->fc_flag & FC_NO_RCV_BUF)) {
               /* IP Response Ring <num> out of posted buffers */
               fc_log_printf_msg_vargs( binfo->fc_brd_no,
                      &fc_msgBlk0602,                   /* ptr to msg structure */
                       fc_mes0602,                      /* ptr to msg */
                        fc_msgBlk0602.msgPreambleStr,   /* begin varargs */
                         rp->fc_ringno,
                          rp->fc_missbufcnt,
                           FCSTATCTR.NoRcvBuf);         /* end varargs */
            }
            binfo->fc_flag |= FC_NO_RCV_BUF;

            fc_post_mbuf(p_dev_ctl, rp, 0);
         }
         else
            NDDSTAT.ndd_ierrors++;
dropout:
         NDDSTAT.ndd_ipackets_drop++;
         fc_free_iocb_buf(p_dev_ctl, rp, savetemp);
         if (p_mbuf) {
            m_freem(p_mbuf);
         }
         return(0);
      }

      if (cmd->ulpBdeCount == 0) {
         temp = (IOCBQ * )temp->q;
         continue;
      }
      for (i = 0; i < (int)cmd->ulpBdeCount; i++) {
         matp = fc_getvaddr(p_dev_ctl, rp, (uchar *)
               getPaddr(cmd->un.cont64[i].addrHigh, cmd->un.cont64[i].addrLow));
         if (matp == 0) {
            uchar *bdeAddr;

            bdeAddr = (uchar *)getPaddr(cmd->un.cont64[0].addrHigh,
               cmd->un.cont64[0].addrLow);
            goto dropout;
         }

         mp = (fcipbuf_t * )matp;
         if (last_mp) {
            fcnextdata(last_mp) = mp;
         } else {
            p_mbuf = mp;
         }
         last_mp = mp;
         fcnextdata(mp) = 0;
         fcsetdatalen(mp, cmd->un.cont64[i].tus.f.bdeSize);
         count += cmd->un.cont64[i].tus.f.bdeSize;
      }

      fc_post_mbuf(p_dev_ctl, rp, i);
      cmd->ulpBdeCount = 0;
      temp = (IOCBQ * )temp->q;
   }

   if (p_mbuf == 0) {
      goto dropout;
   }

   binfo->fc_flag &= ~FC_NO_RCV_BUF;

   /* Set any IP buffer flags to indicate a recieve buffer, if needed */

   if (++NDDSTAT.ndd_ipackets_lsw == 0)
      NDDSTAT.ndd_ipackets_msw++;

   NDDSTAT.ndd_ibytes_lsw += count;
   if ((int)NDDSTAT.ndd_ibytes_lsw < count)
      NDDSTAT.ndd_ibytes_msw++;
   nh = (NETHDR * )fcdata(p_mbuf);

   if(lpfc_nethdr == 0) {
      emac_t        * ep;

      /* Adjust mbuf count now */
      count -= 2;

      fcpktlen(p_mbuf) = count; /* total data in mbuf */
      fcincdatalen(p_mbuf, -2);

      fcdata(p_mbuf) += 2;
      ep = (emac_t * )(fcdata(p_mbuf));
      daddr =  (uchar *)ep->dest_addr;
      saddr =  (uchar *)ep->src_addr;
      ep->llc_len = (count - sizeof(emac_t));
   }
   else {
      NETHDR    * np;

      np = (NETHDR * )(fcdata(p_mbuf));
      daddr = np->fc_destname.IEEE;
      saddr = np->fc_srcname.IEEE;
      fcpktlen(p_mbuf) = count; /* total data in mbuf */
   }

   if (count < (HDR_LEN + sizeof(snaphdr_t)))
      goto dropout;

   /* If this is first broadcast received from that address */
   if (savecmd->un.xrseq.w5.hcsw.Fctl & BC) {
bcst:
      FCSTATCTR.frameRcvBcast++;
      if (++NDDSTAT.ndd_ifInBcastPkts_lsw == 0)
         NDDSTAT.ndd_ifInBcastPkts_msw++;

      fc_bcopy((char *)fcbroadcastaddr, (char *)daddr, MACADDR_LEN);

      if ((ndlp = fc_findnode_odid(binfo, NLP_SEARCH_ALL,
          (uint32)savecmd->un.xrseq.xrsqRo)) == 0) {

         /* Need to cache the did / portname */
         if((ndlp = (NODELIST *)fc_mem_get(binfo, MEM_NLP))) {
            fc_bzero((void *)ndlp, sizeof(NODELIST));
            ndlp->sync = binfo->fc_sync;
            ndlp->capabilities = binfo->fc_capabilities;
            ndlp->nlp_DID = savecmd->un.xrseq.xrsqRo;
            fc_bcopy(&nh->fc_srcname, &ndlp->nlp_portname, sizeof(NAME_TYPE));
            ndlp->nlp_state = NLP_LIMBO;
            fc_nlp_bind(binfo, ndlp);
         }
         else {
            goto dropout;
         }
      }
   } else {
      if ((ndlp = binfo->fc_nlplookup[savecmd->ulpIoTag]) == 0) {
         if(nh->fc_destname.IEEE[0] == 0xff) { 
            if((nh->fc_destname.IEEE[1] == 0xff) &&
               (nh->fc_destname.IEEE[2] == 0xff) &&
               (nh->fc_destname.IEEE[3] == 0xff) &&
               (nh->fc_destname.IEEE[4] == 0xff) &&
               (nh->fc_destname.IEEE[5] == 0xff)) {
               goto bcst;
            }
         }
         /* Need to send LOGOUT for this RPI */
         if ((mb = (MAILBOXQ * )fc_mem_get(binfo, MEM_MBOX))) {
            fc_read_rpi(binfo, (uint32)savecmd->ulpIoTag, 
                (MAILBOX * )mb, (uint32)ELS_CMD_LOGO);
            if (issue_mb_cmd(binfo, (MAILBOX * )mb, MBX_NOWAIT) != MBX_BUSY) {
               fc_mem_put(binfo, MEM_MBOX, (uchar * )mb);
            }
         }
         goto dropout;
      }
   }


   if(lpfc_nethdr == 0) {
      fc_bcopy(ndlp->nlp_portname.IEEE, (char *)saddr, MACADDR_LEN);
   }
   if ((p_dev_ctl->device_state != OPENED) || 
       (p_ndd->nd_receive == 0)) {
      goto dropout;
   }
   ndlp->nlp_type |= NLP_IP_NODE;

   unlock_enable(FC_LVL, &CMD_LOCK);
   (*(p_ndd->nd_receive))(p_ndd, p_mbuf, p_dev_ctl);
   i = disable_lock(FC_LVL, &CMD_LOCK);

   return(1);
}       /* End handle_iprcv_seq */

/**********************************************/
/**  handle_elsrcv_seq                       **/
/**                                          **/
/**********************************************/
_static_ int
handle_elsrcv_seq(
fc_dev_ctl_t *p_dev_ctl,
RING         *rp,
IOCBQ        *temp)
{
   FC_BRD_INFO   * binfo;
   IOCB          * cmd = 0;
   IOCB          * savecmd;
   IOCBQ         * savetemp;
   MATCHMAP     * p_mbuf, *last_mp;
   ndd_t         * p_ndd;
   MATCHMAP      * matp;
   uint32          ctx;
   int  count, i, la;

   binfo = &BINFO;
   p_ndd = (ndd_t * ) & (NDD);

   p_mbuf = 0;
   matp = (MATCHMAP *)0; /* prevent compiler warning */

   last_mp = 0;
   count = 0;
   la = 0;

   savetemp = temp;
   if (binfo->fc_ffstate < FC_READY) {
      goto dropout;
   }

   ctx = 0;
   savecmd = &temp->iocb;
   while (temp) {
      cmd = &temp->iocb;
      if(ctx == 0)
         ctx = (uint32)(cmd->ulpContext);
      if (cmd->ulpStatus) {
         if ((cmd->ulpStatus == IOSTAT_LOCAL_REJECT) && 
             ((cmd->un.ulpWord[4] & 0xff) == IOERR_RCV_BUFFER_WAITING)) {
            FCSTATCTR.NoRcvBuf++;
            if(!(binfo->fc_flag & FC_NO_RCV_BUF)) {
               /* Rcv Ring <num> out of posted buffers */
               fc_log_printf_msg_vargs( binfo->fc_brd_no,
                      &fc_msgBlk0603,                   /* ptr to msg structure */
                       fc_mes0603,                      /* ptr to msg */
                        fc_msgBlk0603.msgPreambleStr,   /* begin varargs */
                         rp->fc_ringno,
                          rp->fc_missbufcnt,
                           FCSTATCTR.NoRcvBuf);         /* end varargs */
            }
            binfo->fc_flag |= FC_NO_RCV_BUF;

            fc_post_buffer(p_dev_ctl, rp, 0);
         }
         goto dropout;
      }

      if (cmd->ulpBdeCount == 0) {
         temp = (IOCBQ * )temp->q;
         continue;
      }
      for (i = 0; i < (int)cmd->ulpBdeCount; i++) {
         matp = fc_getvaddr(p_dev_ctl, rp, (uchar *)
               getPaddr(cmd->un.cont64[i].addrHigh, cmd->un.cont64[i].addrLow));
         if (matp == 0) {
            uchar *bdeAddr;

            bdeAddr = (uchar *)getPaddr(cmd->un.cont64[0].addrHigh,
                  cmd->un.cont64[0].addrLow);

            goto dropout;
         }

         /* Typically for Unsolicited CT requests */

         if (last_mp) {
            last_mp->fc_mptr = (void *)matp;
         } else {
            p_mbuf = matp;
         }
         last_mp = matp;
         matp->fc_mptr = 0;
         count += cmd->un.cont64[i].tus.f.bdeSize;
      }

      fc_post_buffer(p_dev_ctl, rp, i);
      cmd->ulpBdeCount = 0;
      temp = (IOCBQ * )temp->q;
   }

   if (p_mbuf == 0) {
      goto dropout;
   }
   binfo->fc_flag &= ~FC_NO_RCV_BUF;
   if(dfc_put_event(p_dev_ctl, FC_REG_CT_EVENT, ctx, (void *)p_mbuf, (void *)((ulong)count))) {
      fc_free_iocb_buf(p_dev_ctl, rp, savetemp);
      return(0);
   }

dropout:
   fc_free_iocb_buf(p_dev_ctl, rp, savetemp);
   while (p_mbuf) {
      matp = p_mbuf;
      p_mbuf = (MATCHMAP *)matp->fc_mptr;
      fc_mem_put(binfo, MEM_BUF, (uchar * )matp);
   }
   return(0);
}       /* End handle_elsrcv_seq */


/**************************************************/
/**  fc_post_buffer                              **/
/**                                              **/
/**  This routine will post count buffers to the **/
/**  ring with the QUE_RING_BUF_CN command. This **/
/**  allows 3 buffers / command to be posted.    **/
/**  Returns the number of buffers NOT posted.   **/
/**************************************************/
_static_ int
fc_post_buffer(
fc_dev_ctl_t *p_dev_ctl,
RING        *rp,
int cnt)
{
   IOCB          * icmd;
   IOCBQ         * temp;
   int  i, j;
   ushort       tag;
   ushort       maxqbuf;
   MATCHMAP      * mp;
   FC_BRD_INFO   * binfo;

   binfo = &BINFO;
   mp = 0;
   if (binfo->fc_flag & FC_SLI2)
      maxqbuf = 2;
   else
      maxqbuf = 3;

   tag = (ushort)cnt;
   cnt += rp->fc_missbufcnt;
   /* While there are buffers to post */
   while (cnt) {
      if ((temp = (IOCBQ * )fc_mem_get(binfo, MEM_IOCB)) == 0) {
         rp->fc_missbufcnt = cnt;
         return(cnt);
      }
      fc_bzero((void *)temp, sizeof(IOCBQ));
      icmd = &temp->iocb;

      /* Max buffers can be posted per command */
      for (i = 0; i < maxqbuf; i++) {
         if (cnt <= 0)
            break;

         /* fill in BDEs for command */
         if ((mp = (MATCHMAP * )fc_mem_get(binfo, MEM_BUF)) == 0) {
            icmd->ulpBdeCount = i;
            for (j = 0; j < i; j++) {
               if (binfo->fc_flag & FC_SLI2) {
                  mp = fc_getvaddr(p_dev_ctl, rp,
                      (uchar * )getPaddr(icmd->un.cont64[j].addrHigh,
                         icmd->un.cont64[j].addrLow));
               }
               else {
                  mp = fc_getvaddr(p_dev_ctl, rp,
                      (uchar * )((ulong)icmd->un.cont[j].bdeAddress));
               }
               if (mp) {
                  fc_mem_put(binfo, MEM_BUF, (uchar * )mp);
               }
            }

            rp->fc_missbufcnt = cnt + i;
            fc_mem_put(binfo, MEM_IOCB, (uchar * )temp);
            return(cnt + i);
         }

         /* map that page and save the address pair for lookup later */
         if (binfo->fc_flag & FC_SLI2) {
            fc_mapvaddr(binfo, rp, mp,
               (uint32 *) & icmd->un.cont64[i].addrHigh,
               (uint32 *) & icmd->un.cont64[i].addrLow);
            icmd->un.cont64[i].tus.f.bdeSize = FCELSSIZE;
            icmd->ulpCommand = CMD_QUE_RING_BUF64_CN;
         } else {
            fc_mapvaddr(binfo, rp, mp,
               0, (uint32 *) & icmd->un.cont[i].bdeAddress);
            icmd->un.cont[i].bdeSize = FCELSSIZE;
            icmd->ulpCommand = CMD_QUE_RING_BUF_CN;
         }
         cnt--;
      }

      icmd->ulpIoTag = tag;
      icmd->ulpBdeCount = i;
      icmd->ulpLe = 1;

      icmd->ulpOwner = OWN_CHIP;

      temp->bp = (uchar * )mp;  /* used for delimiter between commands */


      FCSTATCTR.cmdQbuf++;
      issue_iocb_cmd(binfo, rp, temp);
   }

   rp->fc_missbufcnt = 0;
   return(0);
}       /* End fc_post_buffer */


/**************************************************/
/**  fc_post_mbuf                                **/
/**                                              **/
/**  This routine will post count buffers to the **/
/**  ring with the QUE_RING_BUF_CN command. This **/
/**  allows 3 buffers / command to be posted.    **/
/**  Returns the number of buffers NOT posted.   **/
/**************************************************/
_static_ int
fc_post_mbuf(
fc_dev_ctl_t *p_dev_ctl,
RING        *rp,
int cnt)
{
   FC_BRD_INFO * binfo;
   IOCB          * icmd;
   IOCBQ         * temp;
   int  i, j;
   ushort       tag;
   ushort       maxqbuf;
   fcipbuf_t     * mp;

   binfo = &BINFO;
   mp = 0;
   if (binfo->fc_flag & FC_SLI2)
      maxqbuf = 2;
   else
      maxqbuf = 3;

   tag = (ushort)cnt;
   cnt += rp->fc_missbufcnt;
   /* While there are buffers to post */
   while (cnt) {
      if ((temp = (IOCBQ * )fc_mem_get(binfo, MEM_IOCB)) == 0) {
         rp->fc_missbufcnt = cnt;
         return(cnt);
      }
      fc_bzero((void *)temp, sizeof(IOCBQ));
      icmd = &temp->iocb;

      /* Max buffers can be posted per command */
      for (i = 0; i < maxqbuf; i++) {
         if (cnt <= 0)
            break;

         /* fill in BDEs for command */
         if ((mp = (fcipbuf_t * )m_getclust(M_DONTWAIT, MT_DATA)) == 0) {

out:
            /* Post buffer for IP ring <num> failed */
            fc_log_printf_msg_vargs( binfo->fc_brd_no,
                   &fc_msgBlk0604,                   /* ptr to msg structure */
                    fc_mes0604,                      /* ptr to msg */
                     fc_msgBlk0604.msgPreambleStr,   /* begin varargs */
                      rp->fc_ringno,
                       rp->fc_missbufcnt);           /* end varargs */
            icmd->ulpBdeCount = i;
            for (j = 0; j < i; j++) {
               if (binfo->fc_flag & FC_SLI2) {
                  mp = (fcipbuf_t * )fc_getvaddr(p_dev_ctl, rp,
                      (uchar * )getPaddr(icmd->un.cont64[j].addrHigh,
                         icmd->un.cont64[j].addrLow));
               }
               else {
                  mp = (fcipbuf_t * )fc_getvaddr(p_dev_ctl, rp,
                      (uchar * )((ulong)icmd->un.cont[j].bdeAddress));
               }
               if (mp) {
                  fcnextdata(mp) = 0;
                  fcnextpkt(mp) = 0;
                  m_freem(mp);
               }
            }

            rp->fc_missbufcnt = cnt + i;
            fc_mem_put(binfo, MEM_IOCB, (uchar * )temp);
            return(cnt + i);
         }
         {
         MBUF_INFO     * buf_info;
         MBUF_INFO    bufinfo;

         buf_info = &bufinfo;
         buf_info->virt = (uint32 * )fcdata(mp);
         buf_info->size = fcPAGESIZE;
         buf_info->flags  = (FC_MBUF_PHYSONLY | FC_MBUF_DMA);
         buf_info->align = 0;
         buf_info->dma_handle = 0;

         /* Map page of memory associated with m_data for read/write */
         fc_malloc(p_dev_ctl, buf_info);
         if (buf_info->phys == NULL) {
            /* mapping that page failed */
            goto out;
         }
         fcnextpkt(mp) = (fcipbuf_t * )buf_info->phys;
         fcsethandle(mp, buf_info->dma_handle);
         }
         /* map that page and save the address pair for lookup later */
         if (binfo->fc_flag & FC_SLI2) {
            fc_mapvaddr(binfo, rp, (MATCHMAP * )mp,
                (uint32 *) & icmd->un.cont64[i].addrHigh,
                (uint32 *) & icmd->un.cont64[i].addrLow);
            icmd->un.cont64[i].tus.f.bdeSize = FC_RCV_BUF_SIZE;
            icmd->ulpCommand = CMD_QUE_RING_BUF64_CN;
         } else {
            fc_mapvaddr(binfo, rp, (MATCHMAP * )mp,
                0, (uint32 *) & icmd->un.cont[i].bdeAddress);
            icmd->un.cont[i].bdeSize = FC_RCV_BUF_SIZE;
            icmd->ulpCommand = CMD_QUE_RING_BUF_CN;
         }
         cnt--;
      }

      icmd->ulpIoTag = tag;
      icmd->ulpBdeCount = i;
      icmd->ulpLe = 1;

      icmd->ulpOwner = OWN_CHIP;

      temp->bp = (uchar * )mp;  /* used for delimiter between commands */

      FCSTATCTR.cmdQbuf++;
      issue_iocb_cmd(binfo, rp, temp);
   }

   rp->fc_missbufcnt = 0;
   return(0);
}       /* End fc_post_mbuf */


_static_ int
fc_free_iocb_buf(
fc_dev_ctl_t *p_dev_ctl,
RING        *rp,
IOCBQ       *temp)
{
   FC_BRD_INFO * binfo;
   IOCB        * cmd;
   int  i;
   MATCHMAP    * mp;

   binfo = &BINFO;
   while (temp) {
      cmd = &temp->iocb;
      for (i = 0; i < (int)cmd->ulpBdeCount; i++) {
         if (binfo->fc_flag & FC_SLI2) {
            mp = fc_getvaddr(p_dev_ctl, rp, (uchar * )
               getPaddr(cmd->un.cont64[i].addrHigh, cmd->un.cont64[i].addrLow));
         }
         else {
            mp = fc_getvaddr(p_dev_ctl, rp, (uchar * )((ulong)cmd->un.cont[i].bdeAddress));
         }

         if (mp) {
            if (rp->fc_ringno == FC_ELS_RING) {
               fc_mem_put(binfo, MEM_BUF, (uchar * )mp);
            }
            else if (rp->fc_ringno == FC_IP_RING) {
               fcipbuf_t * mbuf;

               mbuf = (fcipbuf_t * )mp;
               fcnextdata(mbuf) = 0;
               fcnextpkt(mbuf) = 0;
               m_freem(mbuf);
            }
         }
      }
      switch (rp->fc_ringno) {
      case FC_ELS_RING:
         fc_post_buffer(p_dev_ctl, rp, i);
         break;
      case FC_IP_RING:
         fc_post_mbuf(p_dev_ctl, rp, i);
         break;
      }
      temp = (IOCBQ * )temp->q;
   }
   return(0);
}       /* End fc_free_iocb_buf */


/*
 * Returns 0 if pn1 < pn2
 * Returns 1 if pn1 > pn2
 * Returns 2 if pn1 = pn2
 */
_static_ int
fc_geportname(
NAME_TYPE *pn1,
NAME_TYPE *pn2)
{
   int  i;
   uchar * cp1, *cp2;

   i = sizeof(NAME_TYPE);
   cp1 = (uchar * )pn1;
   cp2 = (uchar * )pn2;
   while (i--) {
      if (*cp1 < *cp2) {
         return(0);
      }
      if (*cp1 > *cp2) {
         return(1);
      }
      cp1++;
      cp2++;
   }

   return(2);  /* equal */
}       /* End fc_geportname */


_local_ int
fc_ring_txcnt(
FC_BRD_INFO     *binfo,
int flag)
{
   int  sum = 0;

   if ((binfo->fc_flag & FC_SLI2) && (FCSTATCTR.linkEvent == 0))
      return(0);

   switch (flag) {
   case FC_IP_RING:
      sum += binfo->fc_ring[FC_IP_RING].fc_tx.q_cnt;
      sum += binfo->fc_ring[FC_ELS_RING].fc_tx.q_cnt;
      break;
   case FC_FCP_RING:
      sum += binfo->fc_ring[FC_FCP_RING].fc_tx.q_cnt;
      sum += binfo->fc_ring[FC_ELS_RING].fc_tx.q_cnt;
      break;
   default:
      sum = 1;
      break;
   }
   return(sum);
}       /* End fc_ring_txcnt */


_local_ int
fc_ring_txpcnt(
FC_BRD_INFO     *binfo,
int flag)
{
   int  sum = 0;

   switch (flag) {
   case FC_IP_RING:
      sum += binfo->fc_ring[FC_IP_RING].fc_txp.q_cnt;
      sum += binfo->fc_ring[FC_ELS_RING].fc_txp.q_cnt;
      break;
   case FC_FCP_RING:
      sum += binfo->fc_ring[FC_FCP_RING].fc_txp.q_cnt;
      sum += binfo->fc_ring[FC_ELS_RING].fc_txp.q_cnt;
      break;
   default:
      sum = 1;
      break;
   }
   return(sum);
}       /* End fc_ring_txpcnt */


/*****************************************************************************/
/*
 * NAME:     fc_cmdring_timeout
 *
 * FUNCTION: Fibre Channel driver cmd ring watchdog timer timeout routine.
 *
 * EXECUTION ENVIRONMENT: interrupt only
 *
 * CALLED FROM:
 *      Timer function
 *
 * RETURNS:  
 *      none
 */
/*****************************************************************************/
_static_ void
fc_cmdring_timeout(
fc_dev_ctl_t * p_dev_ctl,
void *l1,
void *l2)
{
   FC_BRD_INFO   * binfo;
   RING  * rp;
   int  i;
   uint32       command;
   uint32        * lp0;
   IOCBQ       * xmitiq;
   IOCBQ       * save;
   IOCB          * icmd;
   MAILBOXQ      * mb;
   MATCHMAP      * mp;
   NODELIST      * ndlp;
   ELS_PKT       * ep;
   fcipbuf_t * p_mbuf;
   fcipbuf_t * m_net;

   if (!p_dev_ctl) {
      return;
   }
   rp = (RING *)l1;
   RINGTMO = 0;

   binfo = &BINFO;
   if ((xmitiq = fc_ringtxp_get(rp, 0)) != NULL) {
      icmd = &xmitiq->iocb;
      switch (icmd->ulpCommand) {
      case CMD_ELS_REQUEST_CR:
      case CMD_ELS_REQUEST64_CR: 
         mp = (MATCHMAP * )xmitiq->bp;
         lp0 = (uint32 * )mp->virt;
         command = *lp0;
         switch (command) {
         case ELS_CMD_FLOGI:    /* Fabric login */
            fc_freenode_did(binfo, Fabric_DID, 1);
            if (binfo->fc_ffstate == FC_FLOGI) {
               binfo->fc_flag &= ~FC_FABRIC;
               if (binfo->fc_topology == TOPOLOGY_LOOP) {
                  binfo->fc_edtov = FF_DEF_EDTOV;
                  binfo->fc_ratov = FF_DEF_RATOV;
                  if ((mb=(MAILBOXQ * )fc_mem_get(binfo, MEM_MBOX | MEM_PRI))) {
                     fc_config_link(p_dev_ctl, (MAILBOX * )mb);
                     if (issue_mb_cmd(binfo, (MAILBOX * )mb, MBX_NOWAIT)
                          != MBX_BUSY) {
                        fc_mem_put(binfo, MEM_MBOX, (uchar * )mb);
                     }
                  }
                  binfo->fc_flag |= FC_DELAY_DISC;
               } else {
                  /* Device Discovery completion error */
                  fc_log_printf_msg_vargs( binfo->fc_brd_no,
                         &fc_msgBlk0220,                   /* ptr to msg structure */
                          fc_mes0220,                      /* ptr to msg */
                           fc_msgBlk0220.msgPreambleStr);  /* begin & end varargs */
                  binfo->fc_ffstate = FC_ERROR;
                  if ((mb=(MAILBOXQ * )fc_mem_get(binfo, MEM_MBOX | MEM_PRI))) {
                     fc_clear_la(binfo, (MAILBOX * )mb);
                     if (issue_mb_cmd(binfo, (MAILBOX * )mb, MBX_NOWAIT)
                          != MBX_BUSY) {
                        fc_mem_put(binfo, MEM_MBOX, (uchar * )mb);
                     }
                  }
               }
            }
            break;

         case ELS_CMD_PLOGI:    /* NPort login */
            if ((ndlp = fc_findnode_odid(binfo, NLP_SEARCH_ALL,
               icmd->un.elsreq.remoteID)) == 0)
               break;

            /* If we are in the middle of Discovery */
            if (ndlp->nlp_action & NLP_DO_DISC_START) {
               /* Goto next entry */
               fc_nextnode(p_dev_ctl, ndlp);
            }
            ndlp->nlp_flag &= ~NLP_REQ_SND;

            fc_freenode(binfo, ndlp, 0);
            ndlp->nlp_state = NLP_LIMBO;
            fc_nlp_bind(binfo, ndlp);
            break;

         case ELS_CMD_PRLI:     /* Process Log In */
            if ((ndlp = fc_findnode_odid(binfo, NLP_SEARCH_ALL,
               icmd->un.elsreq.remoteID)) == 0)
               break;

            /* If we are in the middle of Discovery */
            if (ndlp->nlp_action & NLP_DO_DISC_START) {
               /* Goto next entry */
               fc_nextnode(p_dev_ctl, ndlp);
            }
            ndlp->nlp_flag &= ~NLP_REQ_SND;
            ndlp->nlp_state = NLP_LOGIN;
            fc_nlp_unmap(binfo, ndlp);
            break;

         case ELS_CMD_PDISC:    /* Pdisc */
         case ELS_CMD_ADISC:    /* Adisc */
            if ((ndlp = fc_findnode_odid(binfo, NLP_SEARCH_ALL,
               icmd->un.elsreq.remoteID)) == 0)
               break;

            /* If we are in the middle of Address Authentication */
            if (ndlp->nlp_action & (NLP_DO_ADDR_AUTH)) {
               ndlp->nlp_action &= ~NLP_DO_ADDR_AUTH;
               fc_freenode(binfo, ndlp, 0);
               ndlp->nlp_state = NLP_LIMBO;
               fc_nlp_bind(binfo, ndlp);

               ndlp->nlp_action |= NLP_DO_DISC_START;
               /* Goto next entry */
               fc_nextnode(p_dev_ctl, ndlp);
            } else {
               fc_freenode(binfo, ndlp, 0);
               ndlp->nlp_state = NLP_LIMBO;
               fc_nlp_bind(binfo, ndlp);
            }
            break;

         case ELS_CMD_LOGO:     /* Logout */
            if ((ndlp = fc_findnode_odid(binfo, NLP_SEARCH_ALL,
               icmd->un.elsreq.remoteID)) == 0)
               break;

            /* If we are in the middle of Discovery */
            if (ndlp->nlp_action & NLP_DO_DISC_START) {
               /* Goto next entry */
               fc_nextnode(p_dev_ctl, ndlp);
            }

            fc_freenode(binfo, ndlp, 0);
            ndlp->nlp_state = NLP_LIMBO;
            fc_nlp_bind(binfo, ndlp);
            break;

         case ELS_CMD_FARP:     /* Farp-req */
         case ELS_CMD_FARPR:    /* Farp-res */
            ep = (ELS_PKT * )lp0;
            if((ndlp = fc_findnode_wwpn(binfo, NLP_SEARCH_ALL,
               &ep->un.farp.RportName)) == 0)
               break;
            ndlp->nlp_flag &= ~NLP_FARP_SND;
            ndlp->nlp_action &= ~NLP_DO_ADDR_AUTH;

            /* Check for a FARP generated nlplist entry */
            if (ndlp->nlp_DID == Bcast_DID) {
               fc_freenode(binfo, ndlp, 1);
            }
            break;

         case ELS_CMD_SCR:      /* State Change Registration */
            break;

         default:
            FCSTATCTR.elsCmdPktInval++;
            break;
         }
         if (xmitiq->bp) {
            fc_mem_put(binfo, MEM_BUF, (uchar * )xmitiq->bp);
         }
         if (xmitiq->info) {
            fc_mem_put(binfo, MEM_BUF, (uchar * )xmitiq->info);
         }
         if ((binfo->fc_flag & FC_SLI2) && (xmitiq->bpl)) {
            fc_mem_put(binfo, MEM_BPL, (uchar * )xmitiq->bpl);
         }
         break;

      case CMD_XMIT_ELS_RSP_CX:
      case CMD_XMIT_ELS_RSP64_CX:
         ndlp = (NODELIST * )xmitiq->ndlp;
         /* No retries */
         if ((ndlp) && (ndlp->nlp_flag & NLP_RM_ENTRY) && 
             !(ndlp->nlp_flag & NLP_REQ_SND)) {
            if (ndlp->nlp_type & NLP_FCP_TARGET) {
               ndlp->nlp_flag &= ~NLP_RM_ENTRY;
               ndlp->nlp_action &= ~NLP_DO_ADDR_AUTH;
               if(binfo->fc_ffstate == FC_READY) {
                  if ((ndlp->nlp_flag & NLP_NODEV_TMO) &&
                     (ndlp->nlp_DID != (uint32)0)) {
                     ndlp->nlp_flag |= NLP_NODEV_TMO;
                     if(!(ndlp->nlp_flag & NLP_NS_REMOVED)) {
                        fc_els_cmd(binfo, ELS_CMD_PLOGI,
                           (void *)((ulong)ndlp->nlp_DID), (uint32)0, (ushort)0, ndlp);
                     }
                  }
                  if(!(binfo->fc_flag & FC_RSCN_MODE)) {
                     binfo->fc_flag |= FC_RSCN_MODE;
                     ndlp->nlp_action |= NLP_DO_RSCN;
                     ndlp->nlp_flag &= ~NLP_NODEV_TMO;
                     fc_nextrscn(p_dev_ctl, 1);
                  }
               }
               else {
                  ndlp->nlp_action |= NLP_DO_DISC_START;
                  fc_nextdisc(p_dev_ctl, 1);
               }
            } else
               fc_freenode_did(binfo, ndlp->nlp_DID, 0);
         }
         if (xmitiq->bp) {
            fc_mem_put(binfo, MEM_BUF, (uchar * )xmitiq->bp);
         }
         if ((binfo->fc_flag & FC_SLI2) && (xmitiq->bpl)) {
            fc_mem_put(binfo, MEM_BPL, (uchar * )xmitiq->bpl);
         }
         break;

      case CMD_ABORT_XRI_CN:
         break;

      case CMD_CREATE_XRI_CR:
         break;

      case CMD_XMIT_SEQUENCE_CX:
      case CMD_XMIT_BCAST_CN:
      case CMD_XMIT_SEQUENCE64_CX:
      case CMD_XMIT_BCAST64_CN:
         if (rp->fc_ringno != FC_IP_RING) {
            break;
         }
         NDDSTAT.ndd_xmitque_cur--;
         /* get mbuf ptr for completed xmit */
         m_net = (fcipbuf_t * )xmitiq->bp;
         /* Loop through iocb chain unmap memory pages associated with mbuf */
         if (binfo->fc_flag & FC_SLI2) {
            ULP_BDE64  * bpl;
            MATCHMAP      * bmp;

            bmp = (MATCHMAP * )xmitiq->bpl;
            bpl = (ULP_BDE64 * )bmp->virt;
            while (bpl && xmitiq->iocb.un.xseq64.bdl.bdeSize) {
               fc_bufunmap(p_dev_ctl, (uchar *)getPaddr(bpl->addrHigh, bpl->addrLow), 0, bpl->tus.f.bdeSize);
               bpl++;
               xmitiq->iocb.un.xseq64.bdl.bdeSize -= sizeof(ULP_BDE64);
            }
            fc_mem_put(binfo, MEM_BPL, (uchar * )bmp);
            fc_mem_put(binfo, MEM_IOCB, (uchar * )xmitiq);
            xmitiq = 0;
         } else {
            while (xmitiq) {
               for (i = 0; i < (int)xmitiq->iocb.ulpBdeCount; i++) {
                  fc_bufunmap(p_dev_ctl, (uchar *)((ulong)xmitiq->iocb.un.cont[i].bdeAddress), 0, (uint32)xmitiq->iocb.un.cont[i].bdeSize); 
               }
               save = (IOCBQ * )xmitiq->q;
               fc_mem_put(binfo, MEM_IOCB, (uchar * )xmitiq);
               xmitiq = save;
            }
         }

         /* free mbuf */
         if (m_net) {
            p_mbuf = fcnextdata(m_net);
            fcnextdata(m_net) = 0;
            fcfreehandle(p_dev_ctl, m_net);
            m_freem(m_net);
            if (p_mbuf) {
               fcfreehandle(p_dev_ctl, p_mbuf);
               m_freem(p_mbuf);
            }
         }
         break;

      default:
         break;
      }
      if (xmitiq)
         fc_mem_put(binfo, MEM_IOCB, (uchar * )xmitiq);
      /* Command ring <num> timeout */
      fc_log_printf_msg_vargs( binfo->fc_brd_no,
             &fc_msgBlk1401,                   /* ptr to msg structure */
              fc_mes1401,                      /* ptr to msg */
               fc_msgBlk1401.msgPreambleStr,   /* begin varargs */
                rp->fc_ringno,
                 icmd->ulpCommand );           /* end varargs */
   } else {
      /* Command ring <ringNum> timeout */
      fc_log_printf_msg_vargs( binfo->fc_brd_no,
             &fc_msgBlk1402,                   /* ptr to msg structure */
              fc_mes1402,                      /* ptr to msg */
               fc_msgBlk1402.msgPreambleStr,   /* begin varargs */
                rp->fc_ringno );               /* end varargs */
   }

   if ((rp->fc_ringno == FC_IP_RING) && 
       (binfo->fc_flag & FC_LNK_DOWN)) {
      IOCBQ * xmitiq;

      /* If linkdown, flush out tx and txp queues */
      /* get next command from ring xmit queue */
      while ((xmitiq = fc_ringtx_drain(rp)) != 0) {
         fc_freebufq(p_dev_ctl, rp, xmitiq);
      }

      /* look up xmit next compl */
      while ((xmitiq = fc_ringtxp_get(rp, 0)) != 0) {
         fc_freebufq(p_dev_ctl, rp, xmitiq);
      }
      NDDSTAT.ndd_xmitque_cur = 0;
   }

   return;
}       /* End fc_cmdring_timeout */


/*****************************************************************************/
/*
 * NAME:     fc_linkdown_timeout
 *
 * FUNCTION: Fibre Channel driver link down watchdog timer timeout routine.
 *
 * EXECUTION ENVIRONMENT: interrupt only
 *
 * CALLED FROM:
 *      Timer function
 *
 * RETURNS:  
 *      none
 */
/*****************************************************************************/
_static_ void
fc_linkdown_timeout(
fc_dev_ctl_t * p_dev_ctl,
void *l1,
void *l2)
{
   FC_BRD_INFO   * binfo;
   iCfgParam     * clp;
   RING  * rp;
   NODELIST      * ndlp;
   NODELIST      * new_ndlp;

   if (!p_dev_ctl) {
      return;
   }

   binfo = &BINFO;
   clp = DD_CTL.p_config[binfo->fc_brd_no];
   rp = &binfo->fc_ring[FC_FCP_RING];
   RINGTMO = 0;

   /* EXPIRED linkdown timer */
   fc_log_printf_msg_vargs( binfo->fc_brd_no,
          &fc_msgBlk0750,                   /* ptr to msg structure */
           fc_mes0750,                      /* ptr to msg */
            fc_msgBlk0750.msgPreambleStr,   /* begin varargs */
             (ulong)binfo->fc_ffstate );    /* end varargs */
   if (binfo->fc_ffstate == FC_READY) {
      return;
   }

   if ((binfo->fc_ffstate > FC_LINK_DOWN) &&
      (binfo->fc_ffstate < FC_READY)) {

      /* Set the link down watchdog timer expired flag */
      binfo->fc_flag |= FC_LD_TIMEOUT;
      goto out;
   }

   /* Since the link has been down for so long, call fc_freenode for all
    * SCSI device and clear out all SCSI queues 
    */
   ndlp = binfo->fc_nlpmap_start;
   while(ndlp != (NODELIST *)&binfo->fc_nlpmap_start) {
      new_ndlp = (NODELIST *)ndlp->nlp_listp_next;
      fc_freenode(binfo, ndlp, 0);
      ndlp->nlp_state = NLP_LIMBO;
      fc_nlp_bind(binfo, ndlp);
      ndlp = new_ndlp;
   }

   /* Set the link down watchdog timer expired flag */
   binfo->fc_flag |= FC_LD_TIMEOUT;

   if((clp[CFG_LINKDOWN_TMO].a_current) && (clp[CFG_HOLDIO].a_current == 0)) {
      fc_failio(p_dev_ctl);
   }

out:

   return;
}       /* End fc_linkdown_timeout */


/*****************************************************************************/
/*
 * NAME:     fc_mbox_timeout
 *
 * FUNCTION: Fibre Channel driver mailbox watchdog timer timeout routine.
 *
 * EXECUTION ENVIRONMENT: interrupt only
 *
 * CALLED FROM:
 *      Timer function
 *
 * RETURNS:  
 *      none
 */
/*****************************************************************************/
_static_ void
fc_mbox_timeout(
fc_dev_ctl_t * p_dev_ctl,
void *l1,
void *l2)
{
   FC_BRD_INFO           * binfo;
   MAILBOXQ              * mbox;
   MAILBOX               * swpmb, *mb;
   void *ioa;
   volatile uint32      word0;

   if (!p_dev_ctl) {
      return;
   }

   binfo = &BINFO;
   MBOXTMO = 0;

   binfo->fc_mbox_active = 0;

   if (binfo->fc_flag & FC_SLI2) {
      fc_mpdata_sync(binfo->fc_slim2.dma_handle, 0, 0,
          DDI_DMA_SYNC_FORKERNEL);
      /* First copy command data */
      mb = FC_SLI2_MAILBOX(binfo);
      word0 = *((volatile uint32 * )mb);
      word0 = PCIMEM_LONG(word0);
   } else {
      /* First copy command data */
      ioa = (void *)FC_MAP_MEM(&binfo->fc_iomap_mem);  /* map in SLIM */
      mb = FC_MAILBOX(binfo, ioa);
      word0 = READ_SLIM_ADDR(binfo, ((volatile uint32 * )mb));
      FC_UNMAP_MEMIO(ioa);
   }
   swpmb = (MAILBOX * ) & word0;

   /* Mailbox command <cmd> timeout, status <status.*/
   fc_log_printf_msg_vargs( binfo->fc_brd_no,
          &fc_msgBlk0310,                   /* ptr to msg structure */
           fc_mes0310,                      /* ptr to msg */
            fc_msgBlk0310.msgPreambleStr,   /* begin varargs */
             swpmb->mbxCommand,
              swpmb->mbxStatus);            /* end varargs */
   if ((mbox = fc_mbox_get(binfo))) {
      if (issue_mb_cmd(binfo, (MAILBOX * )mbox, MBX_NOWAIT) != MBX_BUSY) {
         fc_mem_put(binfo, MEM_MBOX, (uchar * )mbox);
      }
   }

   return;
}       /* End fc_mbox_timeout */



/*****************************************************************************/
/*
 * NAME:     fc_fabric_timeout
 *
 * FUNCTION: Fibre Channel driver fabric watchdog timer timeout routine.
 *
 * EXECUTION ENVIRONMENT: interrupt only
 *
 * CALLED FROM:
 *      Timer function
 *
 * RETURNS:  
 *      none
 */
/*****************************************************************************/
_static_ void
fc_fabric_timeout(
fc_dev_ctl_t * p_dev_ctl,
void *l1,
void *l2)
{
   FC_BRD_INFO   * binfo;
   NODELIST      * ndlp;
   NODELIST      * new_ndlp;
   MAILBOXQ      * mb;
   iCfgParam     * clp;

   if (!p_dev_ctl) {
      return;
   }

   binfo = &BINFO;
   FABRICTMO = 0;

   /* Check for wait for FAN timeout */
   if (binfo->fc_ffstate == FC_FLOGI) {
      if((binfo->fc_topology == TOPOLOGY_LOOP) && 
       (binfo->fc_flag & FC_PUBLIC_LOOP)) {
         /* FAN timeout */
         fc_log_printf_msg_vargs( binfo->fc_brd_no,
                &fc_msgBlk0221,                   /* ptr to msg structure */
                 fc_mes0221,                      /* ptr to msg */
                  fc_msgBlk0221.msgPreambleStr);  /* begin & end varargs */
      }
      else {
         /* Initial FLOGI timeout */
         fc_log_printf_msg_vargs( binfo->fc_brd_no,
                &fc_msgBlk0222,                   /* ptr to msg structure */
                 fc_mes0222,                      /* ptr to msg */
                  fc_msgBlk0222.msgPreambleStr);  /* begin & end varargs */
      }

      fc_freenode_did(binfo, Fabric_DID, 1);
      /* FAN timeout, so just do FLOGI instead */
      /* Now build FLOGI payload and issue ELS command */
      fc_els_cmd(binfo, ELS_CMD_FLOGI, (void *)Fabric_DID,
          (uint32)0, (ushort)0, (NODELIST *)0);
      goto out;
   }

   /* Check for wait for NameServer Rsp timeout */
   if (binfo->fc_ffstate == FC_NS_REG) {
      /* NameServer Registration timeout */
      fc_log_printf_msg_vargs( binfo->fc_brd_no,
             &fc_msgBlk0223,                   /* ptr to msg structure */
              fc_mes0223,                      /* ptr to msg */
               fc_msgBlk0223.msgPreambleStr,   /* begin varargs */
                binfo->fc_ns_retry,
                 fc_max_ns_retry);             /* end varargs */
      if ((ndlp = fc_findnode_odid(binfo, NLP_SEARCH_ALL, NameServer_DID))) {
         if(binfo->fc_ns_retry) {
            if(binfo->fc_ns_retry < fc_max_ns_retry) {
               /* Try it one more time */
               if (fc_ns_cmd(p_dev_ctl, ndlp, SLI_CTNS_RFT_ID) == 0) {
                  goto out;
               }
            }
            binfo->fc_ns_retry = 0;
         }
         /* Complete discovery, then issue an INIT_LINK */
         goto ns_tmout;
      }
      goto out;
   }

   /* Check for wait for NameServer Rsp timeout */
   if (binfo->fc_ffstate == FC_NS_QRY) {
      /* NameServer Query timeout */
      fc_log_printf_msg_vargs( binfo->fc_brd_no,
             &fc_msgBlk0224,                   /* ptr to msg structure */
              fc_mes0224,                      /* ptr to msg */
               fc_msgBlk0224.msgPreambleStr,   /* begin varargs */
                binfo->fc_ns_retry,
                 fc_max_ns_retry);             /* end varargs */
      if ((ndlp = fc_findnode_odid(binfo, NLP_SEARCH_ALL, NameServer_DID))) {
         if(binfo->fc_ns_retry) {
            if(binfo->fc_ns_retry < fc_max_ns_retry) {
               /* Try it one more time */
               if (fc_ns_cmd(p_dev_ctl, ndlp, SLI_CTNS_GID_FT) == 0) {
                  goto out;
               }
            }
            binfo->fc_ns_retry = 0;
         }

ns_tmout:
         /* Complete discovery, then issue an INIT_LINK */
         /* This should turn off DELAYED ABTS for ELS timeouts */
         if ((mb = (MAILBOXQ * )fc_mem_get(binfo, MEM_MBOX))) {
            fc_set_slim(binfo, (MAILBOX * )mb, 0x052198, 0);
            if (issue_mb_cmd(binfo, (MAILBOX * )mb, MBX_NOWAIT) != MBX_BUSY) {
               fc_mem_put(binfo, MEM_MBOX, (uchar * )mb);
            }
         }

         /* Nothing to authenticate, so CLEAR_LA right now */
         if ((mb = (MAILBOXQ * )fc_mem_get(binfo, MEM_MBOX | MEM_PRI))) {
            binfo->fc_ffstate = FC_CLEAR_LA;
            fc_clear_la(binfo, (MAILBOX * )mb);
            if (issue_mb_cmd(binfo, (MAILBOX * )mb, MBX_NOWAIT) != MBX_BUSY) {
               fc_mem_put(binfo, MEM_MBOX, (uchar * )mb);
            }
            /* Device Discovery completes */
            fc_log_printf_msg_vargs( binfo->fc_brd_no,
                   &fc_msgBlk0225,                   /* ptr to msg structure */
                    fc_mes0225,                      /* ptr to msg */
                     fc_msgBlk0225.msgPreambleStr);  /* begin & end varargs */
         } else {
            /* Device Discovery completion error */
            fc_log_printf_msg_vargs( binfo->fc_brd_no,
                   &fc_msgBlk0226,                   /* ptr to msg structure */
                    fc_mes0226,                      /* ptr to msg */
                     fc_msgBlk0226.msgPreambleStr);  /* begin & end varargs */
            binfo->fc_ffstate = FC_ERROR;
         }

         binfo->fc_firstopen++;
         if(binfo->fc_firstopen >= fc_max_ns_retry) {
            goto out;
         }
         
         /* Get a buffer to use for the mailbox command */
         if ((mb = (MAILBOXQ * )fc_mem_get(binfo, MEM_MBOX | MEM_PRI))) {
            clp = DD_CTL.p_config[binfo->fc_brd_no];

            /* Setup and issue mailbox INITIALIZE LINK command */
            fc_linkdown(p_dev_ctl);
            fc_init_link(binfo, (MAILBOX * )mb, clp[CFG_TOPOLOGY].a_current,
               clp[CFG_LINK_SPEED].a_current);
            ((MAILBOX *)mb)->un.varInitLnk.lipsr_AL_PA = 0;
            if (issue_mb_cmd(binfo, (MAILBOX * )mb, MBX_NOWAIT) != MBX_BUSY)
               fc_mem_put(binfo, MEM_MBOX, (uchar * )mb);
         }
      }
      goto out;
   }

   /* Check for Node Authentication timeout */
   if (binfo->fc_ffstate == FC_LOOP_DISC) {
      int disc;

      disc = 0;
      /* Node Authentication timeout */
      fc_log_printf_msg_vargs( binfo->fc_brd_no,
             &fc_msgBlk0227,                   /* ptr to msg structure */
              fc_mes0227,                      /* ptr to msg */
               fc_msgBlk0227.msgPreambleStr);  /* begin & end varargs */
      ndlp = binfo->fc_nlpbind_start;
      if(ndlp == (NODELIST *)&binfo->fc_nlpbind_start)
        ndlp = binfo->fc_nlpunmap_start;
      if(ndlp == (NODELIST *)&binfo->fc_nlpunmap_start)
         ndlp = binfo->fc_nlpmap_start;
      while(ndlp != (NODELIST *)&binfo->fc_nlpmap_start) {
         new_ndlp = (NODELIST *)ndlp->nlp_listp_next;

         /* Clean up all nodes marked for authentication */
         if (ndlp->nlp_action & NLP_DO_ADDR_AUTH) {
            ndlp->nlp_action &= ~NLP_DO_ADDR_AUTH;
            fc_freenode(binfo, ndlp, 0); 
            ndlp->nlp_state = NLP_LIMBO;
            fc_nlp_bind(binfo, ndlp);
            if (ndlp->nlp_DID != NameServer_DID) {
               ndlp->nlp_action |= NLP_DO_DISC_START;
               disc++;
            }
         }
         else if (ndlp->nlp_action & NLP_DO_DISC_START) {
            if (ndlp->nlp_DID != NameServer_DID) {
               disc++;
            }
         }
         ndlp = new_ndlp;
         if(ndlp == (NODELIST *)&binfo->fc_nlpbind_start)
           ndlp = binfo->fc_nlpunmap_start;
         if(ndlp == (NODELIST *)&binfo->fc_nlpunmap_start)
            ndlp = binfo->fc_nlpmap_start;
      }
      if(disc) {
         fc_nextdisc(p_dev_ctl, fc_max_els_sent);
      }
      else {
         goto ns_tmout;
      }
      goto out;
   }

   /* Check for Node Discovery timeout */
   if (binfo->fc_ffstate == FC_NODE_DISC) {
      /* Node Discovery timeout */
      fc_log_printf_msg_vargs( binfo->fc_brd_no,
             &fc_msgBlk0228,                   /* ptr to msg structure */
              fc_mes0228,                      /* ptr to msg */
               fc_msgBlk0228.msgPreambleStr);  /* begin & end varargs */
      ndlp = binfo->fc_nlpbind_start;
      if(ndlp == (NODELIST *)&binfo->fc_nlpbind_start)
        ndlp = binfo->fc_nlpunmap_start;
      if(ndlp == (NODELIST *)&binfo->fc_nlpunmap_start)
         ndlp = binfo->fc_nlpmap_start;
      while(ndlp != (NODELIST *)&binfo->fc_nlpmap_start) {
         new_ndlp = (NODELIST *)ndlp->nlp_listp_next;

         /* Clean up all nodes marked for discovery/authentication */
         if (ndlp->nlp_action & (NLP_DO_ADDR_AUTH | NLP_DO_DISC_START)) {
            /* Node Discovery timeout */
            fc_log_printf_msg_vargs( binfo->fc_brd_no,
                   &fc_msgBlk0229,                   /* ptr to msg structure */
                    fc_mes0229,                      /* ptr to msg */
                     fc_msgBlk0229.msgPreambleStr,   /* begin varargs */
                      ndlp->nlp_DID,
                       ndlp->nlp_flag,
                        ndlp->nlp_state,
                         ndlp->nlp_type);            /* end varargs */
            ndlp->nlp_flag &= ~(NLP_REQ_SND | NLP_REG_INP | NLP_REQ_SND_ADISC);
            fc_freenode(binfo, ndlp, 0);
            ndlp->nlp_state = NLP_LIMBO;
            fc_nlp_bind(binfo, ndlp);
         }
         ndlp = new_ndlp;
         if(ndlp == (NODELIST *)&binfo->fc_nlpbind_start)
           ndlp = binfo->fc_nlpunmap_start;
         if(ndlp == (NODELIST *)&binfo->fc_nlpunmap_start)
            ndlp = binfo->fc_nlpmap_start;
      }
      /* Nothing to discover, so CLEAR_LA right now */
      if ((mb = (MAILBOXQ * )fc_mem_get(binfo, MEM_MBOX | MEM_PRI))) {
         binfo->fc_ffstate = FC_CLEAR_LA;
         fc_clear_la(binfo, (MAILBOX * )mb);
         if (issue_mb_cmd(binfo, (MAILBOX * )mb,MBX_NOWAIT) != MBX_BUSY) {
            fc_mem_put(binfo, MEM_MBOX, (uchar * )mb);
         }
      } else {
         /* Device Discovery completion error */
         fc_log_printf_msg_vargs( binfo->fc_brd_no,
                &fc_msgBlk0230,                   /* ptr to msg structure */
                 fc_mes0230,                      /* ptr to msg */
                  fc_msgBlk0230.msgPreambleStr);  /* begin & end varargs */
         binfo->fc_ffstate = FC_ERROR;
      }
      goto out;
   }

   /* Check for RSCN timeout */
   if ((binfo->fc_flag & FC_RSCN_MODE) && (binfo->fc_ffstate == FC_READY)) {

      if(binfo->fc_ns_retry) {
         if ((ndlp = fc_findnode_odid(binfo, NLP_SEARCH_ALL, NameServer_DID))) {
            if(binfo->fc_ns_retry < fc_max_ns_retry) {
               /* Try it one more time */
               if (fc_ns_cmd(p_dev_ctl, ndlp, SLI_CTNS_GID_FT) == 0) {
                  goto out;
               }
            }
         }
      }
      /* RSCN timeout */
      fc_log_printf_msg_vargs( binfo->fc_brd_no,
             &fc_msgBlk0231,                   /* ptr to msg structure */
              fc_mes0231,                      /* ptr to msg */
               fc_msgBlk0231.msgPreambleStr,   /* begin varargs */
                binfo->fc_ns_retry,
                 fc_max_ns_retry );            /* end varargs */
      ndlp = binfo->fc_nlpbind_start;
      if(ndlp == (NODELIST *)&binfo->fc_nlpbind_start)
        ndlp = binfo->fc_nlpunmap_start;
      if(ndlp == (NODELIST *)&binfo->fc_nlpunmap_start)
         ndlp = binfo->fc_nlpmap_start;
      while(ndlp != (NODELIST *)&binfo->fc_nlpmap_start) {
         new_ndlp = (NODELIST *)ndlp->nlp_listp_next;

         /* Clean up all nodes marked for rscn */
         if (ndlp->nlp_action & NLP_DO_RSCN) {
            /* Node RSCN timeout */
            fc_log_printf_msg_vargs( binfo->fc_brd_no,
                   &fc_msgBlk0232,                   /* ptr to msg structure */
                    fc_mes0232,                      /* ptr to msg */
                     fc_msgBlk0232.msgPreambleStr,   /* begin varargs */
                      ndlp->nlp_DID,
                       ndlp->nlp_flag,
                        ndlp->nlp_state,
                         ndlp->nlp_type);            /* end varargs */
            ndlp->nlp_flag &= ~(NLP_REQ_SND | NLP_REG_INP | NLP_REQ_SND_ADISC);
            fc_freenode(binfo, ndlp, 0);
            ndlp->nlp_state = NLP_LIMBO;
            fc_nlp_bind(binfo, ndlp);
         }
         ndlp = new_ndlp;
         if(ndlp == (NODELIST *)&binfo->fc_nlpbind_start)
           ndlp = binfo->fc_nlpunmap_start;
         if(ndlp == (NODELIST *)&binfo->fc_nlpunmap_start)
            ndlp = binfo->fc_nlpmap_start;
      }

      binfo->fc_flag &= ~FC_NLP_MORE;
      binfo->fc_nlp_cnt = 0;
      binfo->fc_ns_retry = 0;
      /* fc_nextrscn(p_dev_ctl, fc_max_els_sent); */
      fc_rlip(p_dev_ctl);
      goto out;
   }

   /* Check for pt2pt link up timeout */
   if ((binfo->fc_flag & FC_PT2PT) && (binfo->fc_ffstate != FC_READY)) {
      /* PT2PT link up timeout */
      fc_log_printf_msg_vargs( binfo->fc_brd_no,
             &fc_msgBlk0233,                   /* ptr to msg structure */
              fc_mes0233,                      /* ptr to msg */
               fc_msgBlk0233.msgPreambleStr);  /* begin & end varargs */
      binfo->fc_ffstate = FC_LINK_UP;
      binfo->fc_flag &= ~(FC_LNK_DOWN | FC_PT2PT | FC_PT2PT_PLOGI | 
          FC_LBIT | FC_RSCN_MODE | FC_NLP_MORE |
          FC_RSCN_DISC_TMR  | FC_RSCN_DISCOVERY);

      binfo->fc_myDID = 0;
      if ((mb = (MAILBOXQ * )fc_mem_get(binfo, MEM_MBOX | MEM_PRI))) {
         binfo->fc_ffstate = FC_CFG_LINK;
         fc_config_link(p_dev_ctl, (MAILBOX * )mb);
         if (issue_mb_cmd(binfo, (MAILBOX * )mb, MBX_NOWAIT) != MBX_BUSY) {
            fc_mem_put(binfo, MEM_MBOX, (uchar * )mb);
         }
      }
      goto out;
   }

out:
   return;
}       /* End fc_fabric_timeout */


/*****************************************************************************/
/*
 * NAME:     fc_delay_timeout
 *
 * FUNCTION: Fibre Channel driver delay watchdog timer timeout routine.
 *
 * EXECUTION ENVIRONMENT: interrupt only
 *
 * CALLED FROM:
 *      Timer function
 *
 * RETURNS:  
 *      none
 */
/*****************************************************************************/
_static_ void
fc_delay_timeout(
fc_dev_ctl_t  * p_dev_ctl,
void *l1,
void *l2)
{
   FC_BRD_INFO   * binfo;
   IOCBQ         * iocbq;
   RING          * rp;
   MATCHMAP      * rmp;
   NODELIST      * ndlp;

   binfo = &BINFO;
   rp = &binfo->fc_ring[FC_ELS_RING];
   iocbq = (IOCBQ *)l1;

   if(((rmp = (MATCHMAP *)iocbq->info) != 0) && 
               ((ndlp = (NODELIST *)rmp->fc_mptr) != 0)) {
      /* Don't send PLOGI if we are already logged in! */
      if(ndlp->nlp_state >= NLP_LOGIN) {
         if(iocbq->bp) {
            fc_mem_put(binfo, MEM_BUF, (uchar * )iocbq->bp);
         }
         if (iocbq->info) {
            fc_mem_put(binfo, MEM_BUF, (uchar * )iocbq->info);
         }
         if (iocbq->bpl) {
            fc_mem_put(binfo, MEM_BPL, (uchar * )iocbq->bpl);
         }

         fc_mem_put(binfo, MEM_IOCB, (uchar * )iocbq);
         return;
      }
   }
   /* Delayxmit ELS command <cmd> timeout */
   fc_log_printf_msg_vargs( binfo->fc_brd_no,
          &fc_msgBlk0136,                         /* ptr to msg structure */
           fc_mes0136,                            /* ptr to msg */
            fc_msgBlk0136.msgPreambleStr,         /* begin varargs */
             iocbq->iocb.ulpCommand,
              iocbq->iocb.ulpIoTag,
               iocbq->retry, 
                iocbq->iocb.un.elsreq.remoteID);  /* end varargs */
   issue_iocb_cmd(binfo, rp, iocbq);

   if (((binfo->fc_flag & FC_RSCN_MODE) && (binfo->fc_ffstate == FC_READY)) ||
       (binfo->fc_ffstate == FC_LOOP_DISC) ||
       (binfo->fc_ffstate == FC_NODE_DISC)) {
      binfo->fc_fabrictmo = (2 * binfo->fc_ratov) +
         ((4 * binfo->fc_edtov) / 1000) + 1;
      if(FABRICTMO) {
         fc_clk_res(p_dev_ctl, binfo->fc_fabrictmo, FABRICTMO);
      }
      else {
         FABRICTMO = fc_clk_set(p_dev_ctl, binfo->fc_fabrictmo,
                            fc_fabric_timeout, 0, 0);
      }
   }

   return;
}

/*****************************************************************************/
/*
 * NAME:     fc_nodev_timeout
 *
 * FUNCTION: Fibre Channel driver FCP device disappearing timeout routine.
 *
 * EXECUTION ENVIRONMENT: interrupt only
 *
 * CALLED FROM:
 *      Timer interrupt
 *
 * INPUT:
 *      tp      - pointer to the timer structure 
 *
 * RETURNS:  
 *      none
 */
/*****************************************************************************/
_static_ void
fc_nodev_timeout(
fc_dev_ctl_t  * p_dev_ctl,
void * np,
void *l2)
{
   node_t       * nodep;
   dvi_t        * dev_ptr;
   FC_BRD_INFO  * binfo;
   iCfgParam    * clp;
   NODELIST     * ndlp;
   RING         * rp;
   IOCBQ        * temp;
   IOCBQ        * nexttemp, *prevtemp;
   IOCB         * cmd;
   unsigned long iflag;

   nodep = (node_t *)np;
   binfo = &BINFO;
   rp = &binfo->fc_ring[FC_FCP_RING];
   if(nodep) {
      uint32 did;
      uint32 pan;
      uint32 sid;
      uint32 rpi;

      clp = DD_CTL.p_config[p_dev_ctl->info.fc_brd_no];
      if(nodep->rpi != 0xfffe)
         rpi = nodep->rpi;
      else
         rpi = 0;

      if((ndlp = nodep->nlp) == 0) {
         /*
          * Find the target from the nlplist based on SCSI ID
          */
         ndlp = fc_findnode_scsid(binfo, NLP_SEARCH_MAPPED, nodep->scsi_id);
      }

      if (ndlp) {
         RING       * rp;
         IOCBQ      * iocbq;

         /* EXPIRED nodev timer */
         fc_log_printf_msg_vargs( binfo->fc_brd_no,
                &fc_msgBlk0751,                     /* ptr to msg structure */
                 fc_mes0751,                        /* ptr to msg */
                  fc_msgBlk0751.msgPreambleStr,     /* begin varargs */
                   (ulong)ndlp,
                    ndlp->nlp_flag,
                     ndlp->nlp_state,
                      ndlp->nlp_DID);               /* end varargs */
         pan = ndlp->id.nlp_pan;
         sid = ndlp->id.nlp_sid;
         ndlp->nlp_flag &= ~NLP_NODEV_TMO;
         did = ndlp->nlp_DID;
         if(ndlp->nlp_Rpi)
            rpi = ndlp->nlp_Rpi;
         if(did == 0) {
            if((ndlp->nlp_type & (NLP_AUTOMAP | NLP_SEED_MASK)) &&
               (ndlp->nlp_state == NLP_LIMBO) && ndlp->nlp_oldDID)
               did = ndlp->nlp_oldDID;

            if (ndlp->nlp_flag & NLP_REQ_SND) {
               /* Look through ELS ring and abort ELS cmd */
               rp = &binfo->fc_ring[FC_ELS_RING];
               iocbq = (IOCBQ * )(rp->fc_txp.q_first);
               while (iocbq) {
                  if(iocbq->iocb.un.elsreq.remoteID == did) {
                     iocbq->retry = 0xff;
                     if((binfo->fc_flag & FC_RSCN_MODE) ||
                        (binfo->fc_ffstate < FC_READY)) {
                        if((ndlp->nlp_state >= NLP_PLOGI) &&
                           (ndlp->nlp_state <= NLP_PRLI)) {
                           ndlp->nlp_action &= ~NLP_DO_RSCN;
                           binfo->fc_nlp_cnt--;
                           if ((ndlp->nlp_type & NLP_IP_NODE) && ndlp->nlp_bp) {
                              m_freem((fcipbuf_t *)ndlp->nlp_bp);
                              ndlp->nlp_bp = (uchar * )0;
                           }
                        }
                     }
                  }
                  iocbq = (IOCBQ * )iocbq->q;
               }
               /* In case its on fc_delay_timeout list */
               fc_abort_delay_els_cmd(p_dev_ctl, ndlp->nlp_DID);
            }
         }
         fc_freenode(binfo, ndlp, 0);
         ndlp->nlp_state = NLP_LIMBO;
         fc_nlp_bind(binfo, ndlp);
      } else {
         did = 0;
         pan = 0;
         sid = 0;
      }
      /* Device disappeared, nodev timeout */
      fc_log_printf_msg_vargs( binfo->fc_brd_no,
             &fc_msgBlk0752,                         /* ptr to msg structure */
              fc_mes0752,                            /* ptr to msg */
               fc_msgBlk0752.msgPreambleStr,         /* begin varargs */
                did,
                 sid,
                  pan,
                   clp[CFG_NODEV_TMO].a_current);    /* end varargs */
      nodep->flags |= FC_NODEV_TMO;
      nodep->flags &= ~FC_FCP2_RECOVERY;
      nodep->nodev_tmr = 0;
      for (dev_ptr = nodep->lunlist; dev_ptr != NULL; 
          dev_ptr = dev_ptr->next) {

          /* UNREG_LOGIN from freenode should abort txp I/Os */
          if(ndlp == 0) {
               /* First send ABTS on outstanding I/Os in txp queue */
               fc_abort_fcp_txpq(binfo, dev_ptr);
          }

         fc_fail_pendq(dev_ptr, ENXIO, STAT_ABORTED);
         fc_fail_cmd(dev_ptr, ENXIO, STAT_ABORTED);
         fc_return_standby_queue(dev_ptr, ENXIO, STAT_ABORTED);

         /* Call iodone for all the CLEARQ error bufs */
         fc_free_clearq(dev_ptr);
      }
      /*  fail everything on txq that matches rpi */
      iflag = lpfc_q_disable_lock(p_dev_ctl);
      prevtemp = 0;
      temp = (IOCBQ *)rp->fc_tx.q_first;
      while (temp != NULL) {
         nexttemp = (IOCBQ *)temp->q;
         cmd = &temp->iocb;
         if(cmd->ulpContext == rpi) {
            if(prevtemp)
               prevtemp->q = (uchar *)nexttemp;
            else
               rp->fc_tx.q_first = (uchar *)nexttemp;
            if(rp->fc_tx.q_last == (uchar * )temp) {
               rp->fc_tx.q_last =0;
               break;
            }
            cmd->ulpStatus = IOSTAT_LOCAL_REJECT;
            cmd->un.grsp.perr.statLocalError = IOERR_INVALID_RPI;

            lpfc_q_unlock_enable(p_dev_ctl, iflag);
            handle_fcp_event(p_dev_ctl, rp, temp);
            iflag = lpfc_q_disable_lock(p_dev_ctl);

            fc_mem_put(binfo, MEM_IOCB, (uchar * )temp);
         }
         else
            prevtemp = temp;

         if(rp->fc_tx.q_last == (uchar * )temp)
            break;
         temp = nexttemp;
      }
      lpfc_q_unlock_enable(p_dev_ctl, iflag);
   }
}

/*****************************************************************************/
/*
 * NAME:     fc_rscndisc_timeout
 *
 * FUNCTION: Fibre Channel driver RSCN timer timeout routine.
 *
 * EXECUTION ENVIRONMENT: interrupt only
 *
 * CALLED FROM:
 *      Timer function
 *
 * RETURNS:  
 *      none
 */
/*****************************************************************************/
_local_ void
fc_rscndisc_timeout(
fc_dev_ctl_t  * p_dev_ctl,
void *l1,
void *l2)
{
   FC_BRD_INFO   * binfo;

   binfo = &BINFO;
   binfo->fc_flag |= (FC_RSCN_DISC_TMR | FC_RSCN_DISCOVERY);
   /* EXPIRED RSCN disc timer */
   fc_log_printf_msg_vargs( binfo->fc_brd_no,
          &fc_msgBlk0252,                         /* ptr to msg structure */
           fc_mes0252,                            /* ptr to msg */
            fc_msgBlk0252.msgPreambleStr,         /* begin varargs */
             (ulong)binfo->fc_flag );             /* end varargs */
   fc_nextrscn(p_dev_ctl, fc_max_els_sent);
}

_static_ int
fc_free_fcp_txq(
fc_dev_ctl_t  * p_dev_ctl,
uint32 iotag)
{
   FC_BRD_INFO   * binfo;
   RING          * rp;
   IOCBQ         * temp;
   IOCBQ         * prev_temp;
   IOCB          * cmd;
   unsigned long iflag;

   iflag = lpfc_q_disable_lock(p_dev_ctl);
   binfo = &BINFO;
   rp = &binfo->fc_ring[FC_FCP_RING];

   /* Check to see if iotag is still queued on txq */
   prev_temp = 0;
   temp = (IOCBQ *)(rp->fc_tx.q_first);
   while(temp) {
      cmd = &temp->iocb;
      if(iotag == cmd->ulpIoTag) {
         /* A match so dequeue it */
         if(prev_temp) {
            prev_temp->q = temp->q;
         }
         else {
            rp->fc_tx.q_first = (uchar *)(temp->q);
         }
         if(rp->fc_tx.q_last == (uchar * )temp)
            rp->fc_tx.q_last = (uchar * )prev_temp;
         rp->fc_tx.q_cnt--;
         prev_temp = temp;
         temp = (IOCBQ *)(temp->q);
         fc_mem_put(binfo, MEM_IOCB, (uchar * )prev_temp);
         lpfc_q_unlock_enable(p_dev_ctl, iflag);
         return(1);
      }
      prev_temp = temp;
      temp = (IOCBQ *)(temp->q);
   }
   lpfc_q_unlock_enable(p_dev_ctl, iflag);
   return(0);
}       /* End fc_free_fcp_txq */

/*****************************************************************************/
/*
 * NAME:     fc_scsi_timeout
 *
 * FUNCTION: Fibre Channel driver SCSI FCP timeout routine.
 *
 * EXECUTION ENVIRONMENT: interrupt only
 *
 * CALLED FROM:
 *      Timer interrupt
 *
 * INPUT:
 *      tp      - pointer to the timer structure 
 *
 * RETURNS:  
 *      none
 */
/*****************************************************************************/
_static_ void
fc_scsi_timeout(
fc_dev_ctl_t  * p,
void *l1,
void *l2)
{
   fc_dev_ctl_t  * p_dev_ctl;
   FC_BRD_INFO   * binfo;
   iCfgParam     * clp;
   RING          * rp;
   fc_buf_t      * fcptr, *next_fcptr;
   T_SCSIBUF     * sbp;
   struct buf    * clrptr;
   dvi_t         * dev_ptr;
   int  j, ipri, do_rlip;
   uint32       now;

   curtime(&now);

   /*
     * Search through all outstanding SCSI commands for any that timed out
     */
   for (j = 0; j < MAX_FC_BRDS; j++) {
      p_dev_ctl = DD_CTL.p_dev[j];
      if (p_dev_ctl) {
         do_rlip = 0;
         binfo = &BINFO;
         if(binfo->fc_flag & FC_ESTABLISH_LINK) {
           continue;
         }
         clp = DD_CTL.p_config[binfo->fc_brd_no];
         if (clp[CFG_FCP_ON].a_current) {
            rp = &binfo->fc_ring[FC_FCP_RING];

           if(((clp[CFG_LINKDOWN_TMO].a_current == 0) || 
               clp[CFG_HOLDIO].a_current) && (binfo->fc_ffstate != FC_READY)) {
               continue;
            }

            ipri = disable_lock(FC_LVL, &CMD_LOCK);
            fcptr = (fc_buf_t * ) rp->fc_txp.q_first;
            while (fcptr != NULL) {
               next_fcptr = fcptr->fc_fwd;

               if(fcptr->dev_ptr->queue_state == ACTIVE_PASSTHRU) {
                 /* Don't manage PASSTRU CMD HERE */
                 fc_free_fcp_txq(p_dev_ctl, fcptr->iotag);
                 fcptr = next_fcptr;
                 continue;
               } /* end ACTIVE_PASSTHRU management */

               if(ntimercmp(fcptr->timeout, now, < ) && 
                   ntimerisset(&fcptr->timeout)) {

                  {
                  uint32 did;
                  uint32 pan;
                  uint32 sid;

                  dev_ptr = fcptr->dev_ptr;
                  if ((dev_ptr->nodep) && (dev_ptr->nodep->nlp)) {
                     did = dev_ptr->nodep->nlp->nlp_DID;
                     pan = dev_ptr->nodep->nlp->id.nlp_pan;
                     sid = dev_ptr->nodep->nlp->id.nlp_sid;
                  } else {
                     did = 0;
                     pan = 0;
                     sid = 0;
                  }
                  /* SCSI timeout */
                  fc_log_printf_msg_vargs( binfo->fc_brd_no,
                         &fc_msgBlk0754,                     /* ptr to msg structure */
                          fc_mes0754,                        /* ptr to msg */
                           fc_msgBlk0754.msgPreambleStr,     /* begin varargs */
                            did,
                             FC_SCSID(pan, sid) );           /* end varargs */
                  }
                  if (!(fcptr->flags & FCBUF_ABTS2)) {
                     /* Operation timeout, send ABTS for this exchange */
                     if (fc_abort_xri(binfo, fcptr->dev_ptr,
                         fcptr->iotag, ABORT_TYPE_ABTS)) {
                        /* ABTS not sent this time, out of IOCBs */
                        goto skip_rlip;
                     } else {
                        if (fcptr->flags & FCBUF_ABTS) {
                           /* Second ABTS sent for this command */
                           fcptr->flags |= FCBUF_ABTS2;
                        } else {
                           /* First ABTS sent for this command */
                           fcptr->flags |= FCBUF_ABTS;
                        }
                     }
                     fcptr = next_fcptr;
                     continue;
                  }

                  /* Operation timeout, start loop initialization (LIP) */
                  if (dev_ptr->queue_state != STOPPING) {
                     dev_ptr->queue_state = HALTED;
                  }

                  do_rlip = 1;  

skip_rlip:
                  sbp = fcptr->sc_bufp;
                  fc_deq_fcbuf_active(rp, fcptr->iotag);

                  sbp->bufstruct.b_error = ETIMEDOUT;
                  sbp->bufstruct.b_flags |= B_ERROR;
                  sbp->bufstruct.b_resid = sbp->bufstruct.b_bcount;
                  sbp->status_validity = SC_ADAPTER_ERROR;
              SET_ADAPTER_STATUS(sbp, SC_CMD_TIMEOUT)

                  if (fcptr->fcp_cmd.fcpCntl2) {
                     /* This is a task management command */
                     dev_ptr->ioctl_errno = ETIMEDOUT;

                     if (dev_ptr->ioctl_wakeup == 1) {
                        dev_ptr->ioctl_wakeup = 0;

                        fc_admin_wakeup(p_dev_ctl, dev_ptr, sbp);
                     }
                  } else {
                     /* Don't iodone this buf until adapter cleared out */
                     if(fcptr->flags & FCBUF_INTERNAL) {
                        if(fcptr->fcp_cmd.fcpCdb[0] != FCP_SCSI_REPORT_LUNS) {
                           fc_free(p_dev_ctl, (MBUF_INFO *)sbp);
                        }
                        fc_mem_put(binfo, MEM_IOCB, (uchar * )sbp);

                        if((fcptr->fcp_cmd.fcpCdb[0] == FCP_SCSI_REPORT_LUNS) &&
                          (dev_ptr->nodep) &&
                          (dev_ptr->nodep->rptlunstate == REPORT_LUN_ONGOING)) {
                          dev_ptr->nodep->flags &= ~RETRY_RPTLUN;
                          dev_ptr->nodep->rptlunstate = REPORT_LUN_REQUIRED;
                        }
                     }
                     else {
                        if (p_dev_ctl->timeout_head == NULL)
                           p_dev_ctl->timeout_head = (struct buf *)sbp;
                        else {
                           clrptr = p_dev_ctl->timeout_head;
                           while (clrptr->av_forw)
                              clrptr = clrptr->av_forw;
                           clrptr->av_forw = (struct buf *)sbp;
                        }
                        p_dev_ctl->timeout_count++;
                     }
                     dev_ptr->active_io_count--;
                     dev_ptr->nodep->num_active_io--;
                     sbp->bufstruct.av_forw = NULL;
                  }

                  fc_free_fcp_txq(p_dev_ctl, fcptr->iotag);
                  fc_enq_fcbuf(fcptr);
               }
               fcptr = next_fcptr;
            }
            unlock_enable(ipri, &CMD_LOCK);
         }

         /* fix multiple init_link problem */
         if(do_rlip) {
            ipri = disable_lock(FC_LVL, &CMD_LOCK);
            fc_rlip(p_dev_ctl);
            unlock_enable(ipri, &CMD_LOCK);
         }
         continue;
      }
   }

   SCSI_TMO = fc_clk_set(0, 5, fc_scsi_timeout, 0, 0);
   return;
}       /* End fc_scsi_timeout */

_static_ int
fc_abort_fcp_txpq(
FC_BRD_INFO *binfo,
dvi_t       *dev_ptr)
{
   fc_buf_t      * fcptr1, * next_fcptr;
   RING          * rp;
   int             cnt;
   fc_dev_ctl_t  * p_dev_ctl;
   unsigned long iflag;

   p_dev_ctl = (fc_dev_ctl_t *)binfo->fc_p_dev_ctl;
   iflag = lpfc_q_disable_lock(p_dev_ctl);
   rp = &binfo->fc_ring[FC_FCP_RING];
   cnt = 0;

   /* send ABTS on any outstanding I/O in txp queue */
   fcptr1 = (fc_buf_t *)rp->fc_txp.q_first;
   while (fcptr1 != NULL) {
      next_fcptr = fcptr1->fc_fwd;
      if (fcptr1->dev_ptr == dev_ptr) {
         lpfc_q_unlock_enable(p_dev_ctl, iflag);
         fc_abort_xri(binfo, fcptr1->dev_ptr, fcptr1->iotag, ABORT_TYPE_ABTS);
         iflag = lpfc_q_disable_lock(p_dev_ctl);
         cnt++;
      }
      fcptr1 = next_fcptr;
   }
   lpfc_q_unlock_enable(p_dev_ctl, iflag);
   return(cnt);
}

/*
 * Issue an ABORT_XRI_CN iocb command to abort an FCP command already issued.
 */
_static_ int
fc_abort_xri(
FC_BRD_INFO *binfo,
dvi_t       *dev_ptr,
ushort      iotag,
int flag)
{
   IOCB          * icmd;
   IOCBQ         * temp;
   RING  * rp;

   rp = &binfo->fc_ring[FC_FCP_RING];

   if ((binfo->fc_ffstate != FC_READY) ||
       (dev_ptr->nodep->rpi == 0xfffe)) {
      return(1);
   }

   /* Get an iocb buffer */
   if ((temp = (IOCBQ * )fc_mem_get(binfo, MEM_IOCB)) == 0) {
      return(1);
   }
   fc_bzero((void *)temp, sizeof(IOCBQ));
   icmd = &temp->iocb;

   icmd->un.acxri.abortType = flag;
   icmd->un.acxri.abortContextTag = dev_ptr->nodep->rpi;
   icmd->un.acxri.abortIoTag = iotag;

   /* set up an iotag using special ABTS iotags */
   icmd->ulpIoTag = (unsigned)rp->fc_bufcnt++;
   if (rp->fc_bufcnt == 0) {
      rp->fc_bufcnt = MAX_FCP_CMDS;
   }

   icmd->ulpLe = 1;
   icmd->ulpClass = (dev_ptr->nodep->nlp->id.nlp_fcp_info & 0x0f);
   icmd->ulpCommand = CMD_ABORT_XRI_CN;
   icmd->ulpOwner = OWN_CHIP;

   issue_iocb_cmd(binfo, rp, temp);

   return(0);
}       /* End fc_abort_xri */


/*
 * Issue an ABORT_XRI_CX iocb command to abort an IXri.
 */
_static_ int
fc_abort_ixri_cx(
FC_BRD_INFO *binfo,
ushort      xri,
uint32      cmd,
RING        *rp)
{
   IOCB        * icmd;
   IOCBQ       * temp;
   NODELIST    * ndlp;

   /* Get an iocb buffer */
   if ((temp = (IOCBQ * )fc_mem_get(binfo, MEM_IOCB)) == 0) {
      return(1);
   }

   if( (ndlp = fc_findnode_oxri(binfo, NLP_SEARCH_MAPPED | NLP_SEARCH_UNMAPPED, xri)) == 0 ) {
      fc_mem_put(binfo, MEM_IOCB, (uchar * )temp);
      return (1);
   }

   fc_bzero((void *)temp, sizeof(IOCBQ));
   icmd = &temp->iocb;

   icmd->un.acxri.abortType = ABORT_TYPE_ABTS;
   icmd->ulpContext = xri;

   /* set up an iotag */
   icmd->ulpIoTag0 = (unsigned)rp->fc_iotag++;
   if ((rp->fc_iotag & 0x3fff) == 0) {
      rp->fc_iotag = 1;
   }

   icmd->ulpLe = 1;
   icmd->ulpClass = ndlp->id.nlp_ip_info;
   icmd->ulpCommand = cmd;
   icmd->ulpOwner = OWN_CHIP;

   issue_iocb_cmd(binfo, rp, temp);

   return(0);
}       /* End fc_abort_ixri_cx */


/**************************************************/
/**  handle_mb_cmd                               **/
/**                                              **/
/**  Description: Process a Mailbox Command.     **/
/**  Called from host_interrupt to process MBATT **/
/**                                              **/
/**    Returns:                                  **/
/**                                              **/
/**************************************************/
_static_ int
handle_mb_cmd(
fc_dev_ctl_t *p_dev_ctl,
MAILBOX *mb,
uint32  cmd)
{
   FC_BRD_INFO   * binfo;
   iCfgParam     * clp;
   MAILBOXQ      * mbox;
   NODELIST      * ndlp;
   NODELIST      * new_ndlp;
   struct buf *bp, *nextbp;
   RING    * rp;
   int  i;
   void *ioa;
   uint32   control, ldid, lrpi, ldata;
   node_t  * node_ptr;

   binfo = &BINFO;
   clp = DD_CTL.p_config[binfo->fc_brd_no];

   /* Mailbox command completed successfully, process completion */
   switch (cmd) {
   case MBX_LOAD_SM:
   case MBX_READ_NV:            /* a READ NVPARAMS command completed */
   case MBX_WRITE_NV:           /* a WRITE NVPARAMS command completed */
   case MBX_RUN_BIU_DIAG:
   case MBX_INIT_LINK:          /* a LINK INIT command completed */
   case MBX_SET_SLIM:
   case MBX_SET_DEBUG:
   case MBX_PART_SLIM:          /* a PARTITION SLIM command completed */
   case MBX_CONFIG_RING:        /* a CONFIGURE RING command completed */
   case MBX_RESET_RING:
   case MBX_READ_CONFIG:
   case MBX_READ_RCONFIG:
   case MBX_READ_STATUS:
   case MBX_READ_XRI:
   case MBX_READ_REV:
   case MBX_UNREG_D_ID:
   case MBX_READ_LNK_STAT:
   case MBX_DUMP_MEMORY:
   case MBX_LOAD_AREA:
      break;

   case MBX_CONFIG_LINK:        /* a CONFIGURE LINK command completed */
      /* Change the cmdring_timeout value for IP and ELS commands */
      rp = &binfo->fc_ring[FC_ELS_RING];
      rp->fc_ringtmo = (2 * binfo->fc_ratov) + ((4 * binfo->fc_edtov) / 1000) + 1;
      rp = &binfo->fc_ring[FC_IP_RING];
      rp->fc_ringtmo = (2 * binfo->fc_ratov) + ((4 * binfo->fc_edtov) / 1000) + 1;
      binfo->fc_fabrictmo = (2 * binfo->fc_ratov) + ((4 * binfo->fc_edtov) / 1000) + 1;

      if (binfo->fc_ffstate == FC_CFG_LINK) {
         binfo->fc_ffstate = FC_FLOGI;
         if (binfo->fc_topology == TOPOLOGY_LOOP) {
            /* If we are public loop and L bit was set */
            if ((binfo->fc_flag & FC_PUBLIC_LOOP) && 
                !(binfo->fc_flag & FC_LBIT)) {
               /* Need to wait for FAN - use fabric timer for timeout.
                */
               binfo->fc_fabrictmo = ((binfo->fc_edtov) / 1000) + 1;
               if(FABRICTMO) {
                  fc_clk_res(p_dev_ctl, binfo->fc_fabrictmo, FABRICTMO);
               }
               else {
                  FABRICTMO = fc_clk_set(p_dev_ctl, binfo->fc_fabrictmo,
                            fc_fabric_timeout, 0, 0);
               }
               break;
            }

            binfo->fc_fabrictmo = (2*(binfo->fc_edtov) / 1000) + 1;
            if(FABRICTMO) {
               fc_clk_res(p_dev_ctl, binfo->fc_fabrictmo, FABRICTMO);
            }
            else {
               FABRICTMO = fc_clk_set(p_dev_ctl, binfo->fc_fabrictmo,
                         fc_fabric_timeout, 0, 0);
            }
            /* For power_up == 0 see fc_ffinit */
            if(p_dev_ctl->power_up)
               fc_initial_flogi(p_dev_ctl);
         }
         else {  /* pt2pt */
            /* For power_up == 0 see fc_ffinit */
            if(p_dev_ctl->power_up)
               fc_initial_flogi(p_dev_ctl);
         }
      } else {
         if (binfo->fc_flag & FC_DELAY_DISC) {
            /* Config_link is done, so start discovery */
            binfo->fc_flag &= ~FC_DELAY_DISC;
            fc_discovery(p_dev_ctl);
            if (binfo->fc_flag & FC_FABRIC) {
               /* Register with Fabric for receiving RSCNs */
               fc_els_cmd(binfo, ELS_CMD_SCR, (void *)SCR_DID,
                   (uint32)0, (ushort)0, (NODELIST *)0);
            }
         }
      }
      break;

   case MBX_READ_SPARM: /* a READ SPARAM command completed */
   case MBX_READ_SPARM64:       /* a READ SPARAM command completed */
      {
         MATCHMAP * mp;

         mp = (MATCHMAP * )binfo->fc_mbbp;

         if(mp) {
            fc_mpdata_sync(mp->dma_handle, 0, sizeof(SERV_PARM),
                DDI_DMA_SYNC_FORKERNEL);
            fc_mpdata_outcopy(p_dev_ctl, mp, (uchar * ) & binfo->fc_sparam,
                sizeof(SERV_PARM));

            fc_mem_put(binfo, MEM_BUF, (uchar * )mp);

            binfo->fc_mbbp = 0;


            fc_bcopy((uchar * ) & binfo->fc_sparam.nodeName, (uchar * ) & binfo->fc_nodename,
                sizeof(NAME_TYPE));
            fc_bcopy((uchar * ) & binfo->fc_sparam.portName, (uchar * ) & binfo->fc_portname,
                sizeof(NAME_TYPE));
            fc_bcopy(binfo->fc_portname.IEEE, p_dev_ctl->phys_addr, 6);
         }
         break;
      }

   case MBX_READ_RPI:
   case MBX_READ_RPI64:
      if (binfo->fc_flag & FC_SLI2) {
         /* First copy command data */
         mb = FC_SLI2_MAILBOX(binfo);
         ldata = mb->un.varWords[0]; /* get rpi */
         ldata = PCIMEM_LONG(ldata);
         lrpi = ldata & 0xffff;
         ldata = mb->un.varWords[1]; /* get did */
         ldata = PCIMEM_LONG(ldata);
         ldid = ldata & Mask_DID;
         ldata = mb->un.varWords[30];
         ldata = PCIMEM_LONG(ldata);
      } else {
         /* First copy command data */
         ioa = (void *)FC_MAP_MEM(&binfo->fc_iomap_mem);  /* map in SLIM */
         mb = FC_MAILBOX(binfo, ioa);
         ldata = READ_SLIM_ADDR(binfo, (uint32 * ) & mb->un.varWords[0]);
         lrpi = ldata & 0xffff;
         ldata = READ_SLIM_ADDR(binfo, (uint32 * ) & mb->un.varWords[1]);
         ldid = ldata & Mask_DID;
         ldata = READ_SLIM_ADDR(binfo, (uint32 * ) & mb->un.varWords[30]);
         FC_UNMAP_MEMIO(ioa);
      }

      if (ldata == ELS_CMD_LOGO) {
         if (((ndlp = fc_findnode_odid(binfo, NLP_SEARCH_ALL, ldid)) == 0) || 
             (!(ndlp->nlp_action & NLP_DO_ADDR_AUTH) &&
              !(ndlp->nlp_flag & (NLP_FARP_SND | NLP_REQ_SND)))) {

            if (ndlp) {
               if (ndlp->nlp_Rpi)
                  break;         /* Now we have an rpi so don't logout */
            }
            fc_els_cmd(binfo, ELS_CMD_LOGO, (void *)((ulong)ldid),
                (uint32)0, (ushort)0, ndlp);
         }
      }
      break;

   case MBX_REG_LOGIN:
   case MBX_REG_LOGIN64:
      if (binfo->fc_mbbp) {
         fc_mem_put(binfo, MEM_BUF, (uchar * )binfo->fc_mbbp);
         binfo->fc_mbbp = 0;
      }

      if (binfo->fc_flag & FC_SLI2) {
         /* First copy command data */
         mb = FC_SLI2_MAILBOX(binfo);
         ldata = mb->un.varWords[0]; /* get rpi */
         ldata = PCIMEM_LONG(ldata);
         lrpi = ldata & 0xffff;
         ldata = mb->un.varWords[1]; /* get did */
         ldata = PCIMEM_LONG(ldata);
         ldid = ldata & Mask_DID;
         ldata = mb->un.varWords[30];
         ldata = PCIMEM_LONG(ldata);
      } else {
         /* First copy command data */
         ioa = (void *)FC_MAP_MEM(&binfo->fc_iomap_mem);  /* map in SLIM */
         mb = FC_MAILBOX(binfo, ioa);
         ldata = READ_SLIM_ADDR(binfo, (uint32 * ) & mb->un.varWords[0]);
         lrpi = ldata & 0xffff;
         ldata = READ_SLIM_ADDR(binfo, (uint32 * ) & mb->un.varWords[1]);
         ldid = ldata & Mask_DID;
         ldata = READ_SLIM_ADDR(binfo, (uint32 * ) & mb->un.varWords[30]);
         FC_UNMAP_MEMIO(ioa);
      }

      /* Register RPI, will fill in XRI later */
      if ((ndlp=fc_findnode_odid(binfo, NLP_SEARCH_ALL, ldid))) {
         ndlp->nlp_Rpi = (short)lrpi;
         binfo->fc_nlplookup[lrpi] = ndlp;
         ndlp->nlp_state = NLP_LOGIN;
         /* REG_LOGIN cmpl */
         fc_log_printf_msg_vargs( binfo->fc_brd_no,
                &fc_msgBlk0311,                         /* ptr to msg structure */
                 fc_mes0311,                            /* ptr to msg */
                  fc_msgBlk0311.msgPreambleStr,         /* begin varargs */
                   ndlp->nlp_DID,
                    ndlp->nlp_state,
                     ndlp->nlp_flag,
                      ndlp->nlp_Rpi );                  /* end varargs */
         fc_nlp_unmap(binfo, ndlp);

         /* If word30 is set, send back ACC */
         if (ldata) {
            REG_WD30 wd30;

            wd30.word = ldata;

            /* Wait for ACC to complete before issuing PRLI */
            fc_els_rsp(binfo, ELS_CMD_ACC, (uint32)wd30.f.xri,
                (uint32)wd30.f.class, (void *)0, (uint32)sizeof(SERV_PARM), ndlp);
            ndlp->nlp_flag |= NLP_REG_INP;
            break;
         }

         if (ndlp->nlp_DID == binfo->fc_myDID) {
            ndlp->nlp_state = NLP_LOGIN;
         } else {
            fc_process_reglogin(p_dev_ctl, ndlp);
         }
      } else {
         if (ldata) {
            /* Dropping ELS rsp */
            fc_log_printf_msg_vargs( binfo->fc_brd_no,
                   &fc_msgBlk0103,                     /* ptr to msg structure */
                    fc_mes0103,                        /* ptr to msg */
                     fc_msgBlk0103.msgPreambleStr,     /* begin varargs */
                      ldata,
                       ldid );                         /* end varargs */
         }

         /* Can't find NODELIST entry for this login, so unregister it */
         if ((mbox = (MAILBOXQ * )fc_mem_get(binfo, MEM_MBOX))) {
            fc_unreg_login(binfo, lrpi, (MAILBOX * )mbox);
            if (issue_mb_cmd(binfo, (MAILBOX * )mbox, MBX_NOWAIT) != MBX_BUSY) {
               fc_mem_put(binfo, MEM_MBOX, (uchar * )mbox);
            }
         }
      }

      break;

   case MBX_UNREG_LOGIN:
      if (binfo->fc_flag & FC_SLI2) {
         /* First copy command data */
         mb = FC_SLI2_MAILBOX(binfo);
         ldata = mb->un.varWords[0]; /* get rpi */
         ldata = PCIMEM_LONG(ldata);
         lrpi = ldata & 0xffff;
         ldata = mb->un.varWords[30];
         ldata = PCIMEM_LONG(ldata);
      } else {
         /* First copy command data */
         ioa = (void *)FC_MAP_MEM(&binfo->fc_iomap_mem);  /* map in SLIM */
         mb = FC_MAILBOX(binfo, ioa);
         ldata = READ_SLIM_ADDR(binfo, (uint32 * ) & mb->un.varWords[0]);
         lrpi = ldata & 0xffff;
         ldata = READ_SLIM_ADDR(binfo, (uint32 * ) & mb->un.varWords[30]);
         FC_UNMAP_MEMIO(ioa);
      }

      /* If word30 is set, send back LOGO */
      if (ldata) {
         fc_els_cmd(binfo, ELS_CMD_LOGO, (void *)((ulong)ldata), (uint32)0, (ushort)1, (NODELIST *)0);
      }
      break;

   case MBX_READ_LA:
   case MBX_READ_LA64:
      {
         READ_LA_VAR la;
         MATCHMAP * mp;

         if (binfo->fc_flag & FC_SLI2) {
            /* First copy command data */
            mb = FC_SLI2_MAILBOX(binfo);
            fc_pcimem_bcopy((uint32 * )((char *)mb + sizeof(uint32)), (uint32 * ) & la,
                sizeof(READ_LA_VAR));
         } else {
            /* First copy command data */
            ioa = (void *)FC_MAP_MEM(&binfo->fc_iomap_mem);  /* map in SLIM */
            mb = FC_MAILBOX(binfo, ioa);
            READ_SLIM_COPY(binfo, (uint32 * ) & la,
                (uint32 * )((char *)mb + sizeof(uint32)),
                (sizeof(READ_LA_VAR) / sizeof(uint32)));
            FC_UNMAP_MEMIO(ioa);
         }

         mp = (MATCHMAP * )binfo->fc_mbbp;
         if(mp) {
            fc_mpdata_sync(mp->dma_handle, 0, 128, DDI_DMA_SYNC_FORKERNEL);
            fc_mpdata_outcopy(p_dev_ctl, mp, (uchar * )binfo->alpa_map, 128);

            binfo->fc_mbbp = 0;
            fc_mem_put(binfo, MEM_BUF, (uchar * )mp);
         }

         if (la.pb)
            binfo->fc_flag |= FC_BYPASSED_MODE;
         else
            binfo->fc_flag &= ~FC_BYPASSED_MODE;

         if (((binfo->fc_eventTag + 1) < la.eventTag) ||
            (binfo->fc_eventTag == la.eventTag)) {
            FCSTATCTR.LinkMultiEvent++;
            if (la.attType == AT_LINK_UP) {
               if (binfo->fc_eventTag != 0) {  /* Pegasus */
                  fc_linkdown(p_dev_ctl);
                  if (!(binfo->fc_flag & FC_LD_TIMER)) {
                     /* Start the link down watchdog timer until CLA done */
                     rp = &binfo->fc_ring[FC_FCP_RING];
                     RINGTMO = fc_clk_set(p_dev_ctl, rp->fc_ringtmo,
                         fc_linkdown_timeout, 0, 0);
                     if((clp[CFG_LINKDOWN_TMO].a_current == 0) ||
                        clp[CFG_HOLDIO].a_current) {
                        binfo->fc_flag |= FC_LD_TIMEOUT;
                     }
                     binfo->fc_flag |= FC_LD_TIMER;
                  }
               }
            }
         }

         binfo->fc_eventTag = la.eventTag;

         if (la.attType == AT_LINK_UP) {
            FCSTATCTR.LinkUp++;
            /* Link Up Event received */
            fc_log_printf_msg_vargs( binfo->fc_brd_no,
                   &fc_msgBlk1304,                     /* ptr to msg structure */
                    fc_mes1304,                        /* ptr to msg */
                     fc_msgBlk1304.msgPreambleStr,     /* begin varargs */
                      la.eventTag,
                       binfo->fc_eventTag,
                        la.granted_AL_PA,
                         binfo->alpa_map[0] );         /* end varargs */
            if(clp[CFG_NETWORK_ON].a_current) {
               /* Stop the link down watchdog timer */
               rp = &binfo->fc_ring[FC_IP_RING];
               if(RINGTMO) {
                  fc_clk_can(p_dev_ctl, RINGTMO);
                  RINGTMO = 0;
               }
            }
            binfo->fc_ffstate = FC_LINK_UP;
            binfo->fc_flag &= ~(FC_LNK_DOWN | FC_PT2PT | FC_PT2PT_PLOGI | 
                FC_LBIT | FC_RSCN_MODE | FC_NLP_MORE | FC_DELAY_DISC |
                FC_RSCN_DISC_TMR  | FC_RSCN_DISCOVERY);
            binfo->fc_ns_retry = 0;

            if( la.UlnkSpeed == LA_2GHZ_LINK)
               binfo->fc_linkspeed = LA_2GHZ_LINK;
            else
               binfo->fc_linkspeed = 0;

            if ((binfo->fc_topology = la.topology) == TOPOLOGY_LOOP) {

               if (la.il) {
                  binfo->fc_flag |= FC_LBIT;
                  fc_freenode_did(binfo, Fabric_DID, 1);
               }

               binfo->fc_myDID = la.granted_AL_PA;

               dfc_hba_put_event(p_dev_ctl, HBA_EVENT_LINK_UP, binfo->fc_myDID,
                  la.topology, la.lipType, la.UlnkSpeed);
               dfc_put_event(p_dev_ctl, FC_REG_LINK_EVENT, 0, 0, 0);

               if (binfo->fc_flag & FC_SLI2) {
                  i = la.un.lilpBde64.tus.f.bdeSize;
               } else {
                  i = la.un.lilpBde.bdeSize;
               }
               if (i == 0) {
                  binfo->alpa_map[0] = 0;
               } else {
                  if(clp[CFG_LOG_VERBOSE].a_current & DBG_LINK_EVENT) {
                     int        numalpa, j, k;
                     union {
                        uchar pamap[16];
                        struct {
                           uint32 wd1;
                           uint32 wd2;
                           uint32 wd3;
                           uint32 wd4;
                        } pa;
                     } un;

                     numalpa = binfo->alpa_map[0];
                     j = 0;
                     while (j < numalpa) {
                        fc_bzero(un.pamap, 16);
                        for (k = 1; j < numalpa; k++) {
                           un.pamap[k-1] = binfo->alpa_map[j+1];
                           j++;
                           if (k == 16)
                              break;
                        }
                        /* Link Up Event ALPA map */
                        fc_log_printf_msg_vargs( binfo->fc_brd_no,
                               &fc_msgBlk1305,                 /* ptr to msg structure */
                                fc_mes1305,                    /* ptr to msg */
                                 fc_msgBlk1305.msgPreambleStr, /* begin varargs */
                                  un.pa.wd1,
                                   un.pa.wd2,
                                    un.pa.wd3,
                                     un.pa.wd4 );              /* end varargs */
                     }
                  }
               }
            } else {
               fc_freenode_did(binfo, Fabric_DID, 1);

               binfo->fc_myDID = binfo->fc_pref_DID;

               dfc_hba_put_event(p_dev_ctl, HBA_EVENT_LINK_UP, binfo->fc_myDID,
                  la.topology, la.lipType, la.UlnkSpeed);
               dfc_put_event(p_dev_ctl, FC_REG_LINK_EVENT, 0, 0, 0);
            }

            if ((mbox = (MAILBOXQ * )fc_mem_get(binfo, MEM_MBOX | MEM_PRI))) {
               /* This should turn on DELAYED ABTS for ELS timeouts */
               fc_set_slim(binfo, (MAILBOX * )mbox, 0x052198, 0x1);
               /* unreg_login mailbox command could be executing,
                * queue this command to be processed later.
                */
               fc_mbox_put(binfo, mbox);
            }

            if ((mbox = (MAILBOXQ * )fc_mem_get(binfo, MEM_MBOX | MEM_PRI))) {
               if(fc_read_sparam(p_dev_ctl, (MAILBOX * )mbox) == 0) {
                  /* set_slim mailbox command needs to execute first,
                   * queue this command to be processed later.
                   */
                  fc_mbox_put(binfo, mbox);
               } else {
                  fc_mem_put(binfo, MEM_MBOX, (uchar * )mbox);
               }
            }
            if ((mbox = (MAILBOXQ * )fc_mem_get(binfo, MEM_MBOX | MEM_PRI))) {
               binfo->fc_ffstate = FC_CFG_LINK;
               fc_config_link(p_dev_ctl, (MAILBOX * )mbox);
               /* read_sparam mailbox command needs to execute first,
                * queue this command to be processed later.
                */
               fc_mbox_put(binfo, mbox);
            }


         }      /* end if link up */ 
         else {
            FCSTATCTR.LinkDown++;
            dfc_hba_put_event(p_dev_ctl, HBA_EVENT_LINK_DOWN, binfo->fc_myDID,
               la.topology, la.lipType, la.UlnkSpeed);
            dfc_put_event(p_dev_ctl, FC_REG_LINK_EVENT, 0, 0, 0);
            /* Link Down Event received */
            fc_log_printf_msg_vargs( binfo->fc_brd_no,
                   &fc_msgBlk1306,                     /* ptr to msg structure */
                    fc_mes1306,                        /* ptr to msg */
                     fc_msgBlk1306.msgPreambleStr,     /* begin varargs */
                      la.eventTag,
                       binfo->fc_eventTag,
                        la.granted_AL_PA,
                         binfo->alpa_map[0] );         /* end varargs */
            fc_linkdown(p_dev_ctl);

            if (!(binfo->fc_flag & FC_LD_TIMER)) {
               /* Start the link down watchdog timer until CLA done */
               rp = &binfo->fc_ring[FC_FCP_RING];
               RINGTMO = fc_clk_set(p_dev_ctl, rp->fc_ringtmo,
                      fc_linkdown_timeout, 0, 0);
               if((clp[CFG_LINKDOWN_TMO].a_current == 0) ||
                  clp[CFG_HOLDIO].a_current) {
                  binfo->fc_flag |= FC_LD_TIMEOUT;
               }
               binfo->fc_flag |= FC_LD_TIMER;
            }

            /* turn on Link Attention interrupts - no CLEAR_LA needed */
            binfo->fc_process_LA = 1;
            ioa = (void *)FC_MAP_IO(&binfo->fc_iomap_io);  /* map in io registers */
            control = READ_CSR_REG(binfo, FC_HC_REG(binfo, ioa));
            control |= HC_LAINT_ENA;
            WRITE_CSR_REG(binfo, FC_HC_REG(binfo, ioa), control);
            FC_UNMAP_MEMIO(ioa);
         }
         break;
      }

   case MBX_CLEAR_LA:
      /* Turn on Link Attention interrupts */
      binfo->fc_process_LA = 1;

      ioa = (void *)FC_MAP_IO(&binfo->fc_iomap_io);  /* map in io registers */
      control = READ_CSR_REG(binfo, FC_HC_REG(binfo, ioa));
      control |= HC_LAINT_ENA;
      WRITE_CSR_REG(binfo, FC_HC_REG(binfo, ioa), control);
      FC_UNMAP_MEMIO(ioa);

      if ((!(binfo->fc_flag & FC_LNK_DOWN)) && 
          (binfo->fc_ffstate != FC_ERROR) && 
          (mb->mbxStatus != 0x1601)) {  /* Link is Up */

         if (!(binfo->fc_flag & FC_PT2PT)) {
            /* Device Discovery completes */
            fc_log_printf_msg_vargs( binfo->fc_brd_no,
                   &fc_msgBlk0234,                   /* ptr to msg structure */
                    fc_mes0234,                      /* ptr to msg */
                     fc_msgBlk0234.msgPreambleStr);  /* begin & end varargs */
            binfo->fc_nlp_cnt = 0; /* In case we need to do RSCNs */
            binfo->fc_firstopen = 0;

            /* Fix up any changed RPIs in FCP IOCBs queued up a txq */
            fc_fcp_fix_txq(p_dev_ctl);

            binfo->fc_ffstate = FC_READY;

            /* Check to see if we need to process an RSCN */
            if(binfo->fc_flag & FC_RSCN_MODE) {
               fc_nextrscn(p_dev_ctl, fc_max_els_sent);
            }
         }

         /* Do FDMI to Register HBA and port */ 
         if ((ndlp = fc_findnode_odid(binfo, NLP_SEARCH_ALL, FDMI_DID))) {
           if (fc_fdmi_cmd(p_dev_ctl, ndlp, SLI_MGMT_DPRT)) {
              /* Issue FDMI request failed */
              fc_log_printf_msg_vargs( binfo->fc_brd_no,
                     &fc_msgBlk0219,                     /* ptr to msg structure */
                      fc_mes0219,                        /* ptr to msg */
                       fc_msgBlk0219.msgPreambleStr,     /* begin varargs */
                        SLI_MGMT_DPRT );                 /* end varargs */
           }
         }

         ndlp = binfo->fc_nlpunmap_start;
         if(ndlp == (NODELIST *)&binfo->fc_nlpunmap_start)
            ndlp = binfo->fc_nlpmap_start;
         while(ndlp != (NODELIST *)&binfo->fc_nlpmap_start) {
            new_ndlp = (NODELIST *)ndlp->nlp_listp_next;

            /* skip myself, fabric nodes and partially logged in nodes */
            if ((ndlp->nlp_DID == binfo->fc_myDID) || 
                (ndlp->nlp_type & NLP_FABRIC) || 
                (ndlp->nlp_state != NLP_ALLOC))
               goto loop1;

            /* Allocate exchanges for all IP (non-FCP) nodes */
            if ((ndlp->nlp_Rpi) && 
                (ndlp->nlp_Xri == 0) && 
                ((ndlp->nlp_DID & CT_DID_MASK) != CT_DID_MASK) &&
                !(ndlp->nlp_flag & NLP_RPI_XRI) &&
                !(ndlp->nlp_type & NLP_FCP_TARGET)) {
               ndlp->nlp_flag |= NLP_RPI_XRI;
               fc_create_xri(binfo, &binfo->fc_ring[FC_ELS_RING], ndlp);
            }

            if (ndlp->nlp_type & NLP_FCP_TARGET) {
               int  dev_index;

               dev_index = INDEX(ndlp->id.nlp_pan, ndlp->id.nlp_sid);
               node_ptr =  binfo->device_queue_hash[dev_index].node_ptr;
               if(node_ptr) {
                  /* This is a new device that entered the loop */
                  node_ptr->nlp = ndlp;
                  node_ptr->rpi = ndlp->nlp_Rpi;
                  node_ptr->last_good_rpi = ndlp->nlp_Rpi;
                  node_ptr->scsi_id = dev_index;
                  ndlp->nlp_targetp = (uchar *)node_ptr;
                  node_ptr->flags &= ~FC_NODEV_TMO;
                  ndlp->nlp_flag &= ~NLP_NODEV_TMO;
               }
            }

            if (ndlp->nlp_type & NLP_IP_NODE) {
               fc_restartio(p_dev_ctl, ndlp);
            }
loop1:
            ndlp = new_ndlp;
            if(ndlp == (NODELIST *)&binfo->fc_nlpunmap_start)
               ndlp = binfo->fc_nlpmap_start;
         }

         /* If we are not point to point, reglogin to ourself */
         if (!(binfo->fc_flag & FC_PT2PT)) {
            /* Build nlplist entry and Register login to ourself */
            if ((ndlp = fc_findnode_odid(binfo, NLP_SEARCH_ALL, binfo->fc_myDID))) {
                  ndlp->nlp_DID = binfo->fc_myDID;
                  fc_nlp_logi(binfo, ndlp, &(binfo->fc_portname), &(binfo->fc_nodename));
            }
            else {
               if((ndlp = (NODELIST *)fc_mem_get(binfo, MEM_NLP))) {
                  fc_bzero((void *)ndlp, sizeof(NODELIST));
                  ndlp->sync = binfo->fc_sync;
                  ndlp->capabilities = binfo->fc_capabilities;
                  ndlp->nlp_DID = binfo->fc_myDID;
                  ndlp->nlp_state = NLP_LIMBO;
                  fc_nlp_bind(binfo, ndlp);
                  fc_nlp_logi(binfo, ndlp, &(binfo->fc_portname), &(binfo->fc_nodename));
               }
            }
            if(ndlp) {
               if ((mbox = (MAILBOXQ * )fc_mem_get(binfo, MEM_MBOX | MEM_PRI))){
                  fc_reg_login(binfo, binfo->fc_myDID,
                      (uchar * ) & binfo->fc_sparam, (MAILBOX * )mbox, 0);
                  if (issue_mb_cmd(binfo, (MAILBOX * )mbox, MBX_NOWAIT) != MBX_BUSY) {
                     fc_mem_put(binfo, MEM_MBOX, (uchar * )mbox);
                  }
               }
            }
         } else {
            /* We are pt2pt no fabric */
            if (binfo->fc_flag & FC_PT2PT_PLOGI) {
               /* Build nlplist entry and Register login to ourself */
               if ((ndlp = fc_findnode_odid(binfo, NLP_SEARCH_ALL, binfo->fc_myDID))) {
                  ndlp->nlp_DID = binfo->fc_myDID;
                  fc_nlp_logi(binfo, ndlp, &(binfo->fc_portname), &(binfo->fc_nodename));
               }
               else {
                  if((ndlp = (NODELIST *)fc_mem_get(binfo, MEM_NLP))) {
                     fc_bzero((void *)ndlp, sizeof(NODELIST));
                     ndlp->sync = binfo->fc_sync;
                     ndlp->capabilities = binfo->fc_capabilities;
                     ndlp->nlp_DID = binfo->fc_myDID;
                     ndlp->nlp_state = NLP_LIMBO;
                     fc_nlp_bind(binfo, ndlp);
                     fc_nlp_logi(binfo, ndlp, &(binfo->fc_portname), &(binfo->fc_nodename));
                  }
               }
               if(ndlp) {
                  if ((mbox = (MAILBOXQ * )fc_mem_get(binfo, MEM_MBOX | MEM_PRI))){
                     fc_reg_login(binfo, binfo->fc_myDID,
                         (uchar * ) & binfo->fc_sparam, (MAILBOX * )mbox, 0);
                     if (issue_mb_cmd(binfo, (MAILBOX * )mbox, MBX_NOWAIT) != MBX_BUSY) {
                        fc_mem_put(binfo, MEM_MBOX, (uchar * )mbox);
                     }
                  }
                  fc_els_cmd(binfo, ELS_CMD_PLOGI, (void *)PT2PT_RemoteID,
                      (uint32)0, (ushort)0, (NODELIST *)0);
               }
            }
         }

         if(binfo->fc_flag & FC_ESTABLISH_LINK) {
            binfo->fc_flag &= ~FC_ESTABLISH_LINK;
         }

         if(p_dev_ctl->fc_estabtmo) {
               fc_clk_can(p_dev_ctl, p_dev_ctl->fc_estabtmo);
               p_dev_ctl->fc_estabtmo = 0;
         }

         if(((clp[CFG_LINKDOWN_TMO].a_current == 0) ||
             clp[CFG_HOLDIO].a_current)  &&
             (binfo->fc_flag & FC_LD_TIMEOUT)) {
             fc_failio(p_dev_ctl);
         }

         /* Stop the link down watchdog timer */
         rp = &binfo->fc_ring[FC_FCP_RING];
         if(RINGTMO) {
            fc_clk_can(p_dev_ctl, RINGTMO);
            RINGTMO = 0;
         }
         binfo->fc_flag &= ~(FC_LD_TIMEOUT | FC_LD_TIMER);

         if(clp[CFG_FCP_ON].a_current) {
            fc_restart_all_devices(p_dev_ctl);

            /* Call iodone for any commands that timed out previously */
            for (bp = p_dev_ctl->timeout_head; bp != NULL; ) {
               nextbp = bp->av_forw;
               bp->b_error = ETIMEDOUT;
               bp->b_flags |= B_ERROR;
               bp->b_resid = bp->b_bcount;
               FCSTATCTR.fcpScsiTmo++;
               fc_do_iodone(bp);
               bp = nextbp;
            }
            p_dev_ctl->timeout_count = 0;
            p_dev_ctl->timeout_head = NULL;

            /* Send down any saved FCP commands */
            fc_issue_cmd(p_dev_ctl);
         }

         if (binfo->fc_deferip) {
            handle_ring_event(p_dev_ctl, FC_IP_RING,
                (uint32)binfo->fc_deferip);
            binfo->fc_deferip = 0;
         }
      }
      break;

   default:
      /* Unknown Mailbox command <cmd> completion */
      fc_log_printf_msg_vargs( binfo->fc_brd_no,
             &fc_msgBlk0312,                         /* ptr to msg structure */
              fc_mes0312,                            /* ptr to msg */
               fc_msgBlk0312.msgPreambleStr,         /* begin varargs */
                cmd );                               /* end varargs */
      FCSTATCTR.mboxCmdInval++;
      break;
   }

   binfo->fc_mbbp = 0;
   return(0);
}       /* End handle_mb_cmd */


/**************************************************/
/**  fc_linkdown                                 **/
/**                                              **/
/**  Description: Process a Link Down event.     **/
/**  Called from host_intupt to process LinkDown **/
/**                                              **/
/**    Returns:                                  **/
/**                                              **/
/**************************************************/
_static_ int
fc_linkdown(
fc_dev_ctl_t *p_dev_ctl)
{
   FC_BRD_INFO   * binfo;
   iCfgParam     * clp;
   RING          * rp;
   NODELIST      * ndlp;
   NODELIST      * new_ndlp;
   MAILBOXQ      * mb;
   IOCBQ         * xmitiq;
   IOCBQ         * iocbq;
   MATCHMAP      * mp;
   ULP_BDE64     * addr;

   binfo = &BINFO;
   binfo->fc_prevDID = binfo->fc_myDID;
   binfo->fc_ffstate = FC_LINK_DOWN;
   binfo->fc_flag |= FC_LNK_DOWN;
   binfo->fc_flag &= ~FC_DELAY_PLOGI;


   if ((mb = (MAILBOXQ * )fc_mem_get(binfo, MEM_MBOX | MEM_PRI))) {
      fc_unreg_did(binfo, 0xffffffff, (MAILBOX * )mb);
      if (issue_mb_cmd(binfo, (MAILBOX * )mb, MBX_NOWAIT) != MBX_BUSY) {
         fc_mem_put(binfo, MEM_MBOX, (uchar * )mb);
      }
   }

   /* Free all nodes in nlplist */
   ndlp = binfo->fc_nlpbind_start;
   if(ndlp == (NODELIST *)&binfo->fc_nlpbind_start)
     ndlp = binfo->fc_nlpunmap_start;
   if(ndlp == (NODELIST *)&binfo->fc_nlpunmap_start)
      ndlp = binfo->fc_nlpmap_start;
   while(ndlp != (NODELIST *)&binfo->fc_nlpmap_start) {
      new_ndlp = (NODELIST *)ndlp->nlp_listp_next;

      /* Any RSCNs in progress don't matter at this point */
      ndlp->nlp_action &= ~NLP_DO_RSCN;

      if ((ndlp->nlp_type & NLP_IP_NODE) && ndlp->nlp_bp) {
         m_freem((fcipbuf_t *)ndlp->nlp_bp);
         ndlp->nlp_bp = (uchar * )0;
      }

      /* Need to abort all exchanges, used only on IP */
      if (ndlp->nlp_Xri) {
         fc_rpi_abortxri(binfo, ndlp->nlp_Xri);
         ndlp->nlp_Xri = 0;
      }

      /* Need to free all nodes in the process of login / registration
       * as well as all Fabric nodes and myself.
       */
      if ((ndlp->nlp_DID == binfo->fc_myDID) || 
         (!(ndlp->nlp_type & NLP_FABRIC) && ((ndlp->nlp_DID & CT_DID_MASK) == CT_DID_MASK)) ||
         (binfo->fc_flag & FC_PT2PT) || 
         (ndlp->nlp_state < NLP_ALLOC)) {
         NAME_TYPE   zero_pn;

         fc_bzero((void *)&zero_pn, sizeof(NAME_TYPE));
         if ((fc_geportname(&ndlp->nlp_portname, &zero_pn) == 2) &&
            (ndlp->nlp_state < NLP_LOGIN) &&
            ((ndlp->nlp_type & (NLP_AUTOMAP | NLP_SEED_MASK)) == 0)) {
            fc_freenode(binfo, ndlp, 1);
         }
         else {
            fc_freenode(binfo, ndlp, 0);
            ndlp->nlp_state = NLP_LIMBO;
            fc_nlp_bind(binfo, ndlp);
         }
      }

      /* If we are not using ADISC, free fcp nodes here to avoid excessive
       * actitivity when during PLOGIs when link comes back up.
       */
      clp = DD_CTL.p_config[binfo->fc_brd_no];
      if((ndlp->nlp_state == NLP_ALLOC) &&
         (ndlp->nlp_type & NLP_FCP_TARGET) && 
         ((!clp[CFG_USE_ADISC].a_current))) {
         fc_freenode(binfo, ndlp, 0);
         ndlp->nlp_state = NLP_LIMBO;
         fc_nlp_bind(binfo, ndlp);
      }

      /* Any Discovery in progress doesn't matter at this point */
      ndlp->nlp_action &= ~(NLP_DO_ADDR_AUTH | NLP_DO_DISC_START);

      ndlp = new_ndlp;
      if(ndlp == (NODELIST *)&binfo->fc_nlpbind_start)
        ndlp = binfo->fc_nlpunmap_start;
      if(ndlp == (NODELIST *)&binfo->fc_nlpunmap_start)
         ndlp = binfo->fc_nlpmap_start;
   }

   if (binfo->fc_flag & FC_PT2PT) {
      binfo->fc_myDID = 0;
      if ((mb = (MAILBOXQ * )fc_mem_get(binfo, MEM_MBOX | MEM_PRI))) {
         fc_config_link(p_dev_ctl, (MAILBOX * )mb);
         if (issue_mb_cmd(binfo, (MAILBOX * )mb, MBX_NOWAIT) != MBX_BUSY) {
            fc_mem_put(binfo, MEM_MBOX, (uchar * )mb);
         }
      }
      binfo->fc_flag &= ~(FC_PT2PT | FC_PT2PT_PLOGI);
   }

   clp = DD_CTL.p_config[binfo->fc_brd_no];
   if(clp[CFG_NETWORK_ON].a_current) {
      rp = &binfo->fc_ring[FC_IP_RING];
      /* flush all xmit compls */
      while ((xmitiq = fc_ringtxp_get(rp, 0)) != 0) {
         fc_freebufq(p_dev_ctl, rp, xmitiq);
      }
      NDDSTAT.ndd_xmitque_cur = 0;
   }


   fc_flush_clk_set(p_dev_ctl, fc_delay_timeout);

   if(FABRICTMO) {
      fc_clk_can(p_dev_ctl, FABRICTMO);
      FABRICTMO = 0;
   }

   if(binfo->fc_rscn_disc_wdt) {
      fc_clk_can(p_dev_ctl, binfo->fc_rscn_disc_wdt);
      binfo->fc_rscn_disc_wdt = 0;
   }
   binfo->fc_flag &= ~(FC_RSCN_MODE | FC_RSCN_DISC_TMR  | FC_RSCN_DISCOVERY);
   binfo->fc_rscn_id_cnt = 0;

   /* Free any deferred RSCNs */
   fc_flush_rscn_defer(p_dev_ctl);

   /* Free any delayed ELS xmits */
   fc_abort_delay_els_cmd(p_dev_ctl, 0xffffffff);

   /* Look through ELS ring and remove any ELS cmds in progress */
   rp = &binfo->fc_ring[FC_ELS_RING];
   iocbq = (IOCBQ * )(rp->fc_txp.q_first);
   while (iocbq) {
      iocbq->retry = 0xff;  /* Mark for abort */
      iocbq = (IOCBQ * )iocbq->q;
   }

   if (rp->fc_tx.q_cnt) {
      IOCB    * icmd;
      /* get next command from ring xmit queue */
      xmitiq = fc_ringtx_get(rp);

      while (xmitiq) {
         icmd = &xmitiq->iocb;
         if (icmd->ulpCommand == CMD_IOCB_CONTINUE_CN) {
            fc_mem_put(binfo, MEM_IOCB, (uchar * )xmitiq);
            xmitiq = fc_ringtx_get(rp);
            continue;
         }

         if(xmitiq->bp) {
            fc_mem_put(binfo, MEM_BUF, (uchar * )xmitiq->bp);
         }

         if (binfo->fc_flag & FC_SLI2) {

            mp = (MATCHMAP *)xmitiq->bpl;
            if(mp) {
               addr = (ULP_BDE64 * )mp->virt;
               addr++; /* goto the next one */

               switch (icmd->ulpCommand) {
               case CMD_ELS_REQUEST_CR:
               case CMD_ELS_REQUEST64_CR:
               case CMD_ELS_REQUEST_CX:
               case CMD_ELS_REQUEST64_CX:
                  fc_mem_put(binfo, MEM_BUF, (uchar * )xmitiq->info);
                  break;
               default:
                  if(xmitiq->info)
                     fc_mem_put(binfo, MEM_BUF, (uchar * )xmitiq->info);
               }
               fc_mem_put(binfo, MEM_BPL, (uchar * )mp);
            }
         }
         else {
            if (icmd->un.cont[1].bdeAddress) {
               fc_mem_put(binfo, MEM_BUF, (uchar * )xmitiq->info);
            }
         }
         fc_mem_put(binfo, MEM_IOCB, (uchar * )xmitiq);
         xmitiq = fc_ringtx_get(rp);
      }
   }

   return(0);
}       /* End fc_linkdown */

/**************************************************/
/**  fc_rlip                                     **/
/**                                              **/
/**  Description:                                **/
/**  Called to reset the link with an init_link  **/
/**                                              **/
/**    Returns:                                  **/
/**                                              **/
/**************************************************/
_static_ int
fc_rlip(
fc_dev_ctl_t *p_dev_ctl)
{
   FC_BRD_INFO   * binfo;
   iCfgParam     * clp;
   MAILBOX       * mb;

   binfo = &BINFO;

   /* Start the Fibre Channel reset LIP process */
   if (binfo->fc_ffstate == FC_READY) {
      /* Get a buffer to use for the mailbox command */
      if ((mb = (MAILBOX * )fc_mem_get(binfo, MEM_MBOX | MEM_PRI)) == NULL) {
         /* Device Discovery completion error */
         fc_log_printf_msg_vargs( binfo->fc_brd_no,
                &fc_msgBlk0235,                   /* ptr to msg structure */
                 fc_mes0235,                      /* ptr to msg */
                  fc_msgBlk0235.msgPreambleStr);  /* begin & end varargs */
         binfo->fc_ffstate = FC_ERROR;
         return(1);
      }

      binfo->fc_flag |= FC_SCSI_RLIP;
      clp = DD_CTL.p_config[binfo->fc_brd_no];

      /* Setup and issue mailbox INITIALIZE LINK command */
      fc_linkdown(p_dev_ctl);
      fc_init_link(binfo, (MAILBOX * )mb, clp[CFG_TOPOLOGY].a_current, clp[CFG_LINK_SPEED].a_current);
      mb->un.varInitLnk.lipsr_AL_PA = 0;
      if (issue_mb_cmd(binfo, (MAILBOX * )mb, MBX_NOWAIT) != MBX_BUSY)
         fc_mem_put(binfo, MEM_MBOX, (uchar * )mb);
      /* SCSI Link Reset */
      fc_log_printf_msg_vargs( binfo->fc_brd_no,
             &fc_msgBlk1307,                         /* ptr to msg structure */
              fc_mes1307,                            /* ptr to msg */
               fc_msgBlk1307.msgPreambleStr);        /* begin & end varargs */
   }
   return(0);
}       /* End fc_rlip */

/**************************************************/
/**  fc_ns_cmd                                   **/
/**                                              **/
/**  Description:                                **/
/**  Issue Cmd to NameServer                     **/
/**     SLI_CTNS_GID_FT                          **/
/**     SLI_CTNS_RFT_ID                          **/
/**                                              **/
/**    Returns:                                  **/
/**                                              **/
/**************************************************/
_static_ int
fc_ns_cmd(
fc_dev_ctl_t *p_dev_ctl,
NODELIST        *ndlp,
int cmdcode)
{
   FC_BRD_INFO   * binfo;
   iCfgParam     * clp;
   MATCHMAP      * mp, *bmp;
   SLI_CT_REQUEST * CtReq;
   ULP_BDE64     * bpl;

   binfo = &BINFO;

   /* fill in BDEs for command */
   /* Allocate buffer for command payload */
   if ((mp = (MATCHMAP * )fc_mem_get(binfo, MEM_BUF)) == 0) {
      return(1);
   }

   bmp = 0;

   /* Allocate buffer for Buffer ptr list */
   if ((bmp = (MATCHMAP * )fc_mem_get(binfo, MEM_BPL)) == 0) {
      fc_mem_put(binfo, MEM_BUF, (uchar * )mp);
      return(1);
   }
   bpl = (ULP_BDE64 * )bmp->virt;
   bpl->addrHigh = PCIMEM_LONG((uint32)putPaddrHigh(mp->phys));
   bpl->addrLow = PCIMEM_LONG((uint32)putPaddrLow(mp->phys));
   bpl->tus.f.bdeFlags = 0;
   if (cmdcode == SLI_CTNS_GID_FT)
      bpl->tus.f.bdeSize = GID_REQUEST_SZ;
   else if (cmdcode == SLI_CTNS_RFT_ID)
      bpl->tus.f.bdeSize = RFT_REQUEST_SZ;
   else
      bpl->tus.f.bdeSize = 0;
   bpl->tus.w = PCIMEM_LONG(bpl->tus.w);

   CtReq = (SLI_CT_REQUEST * )mp->virt;
   /* NameServer Req */
   fc_log_printf_msg_vargs( binfo->fc_brd_no,
          &fc_msgBlk0236,                   /* ptr to msg structure */
           fc_mes0236,                      /* ptr to msg */
            fc_msgBlk0236.msgPreambleStr,   /* begin varargs */
             cmdcode,
              binfo->fc_flag,
               binfo->fc_rscn_id_cnt);      /* end varargs */
   fc_bzero((void *)CtReq, sizeof(SLI_CT_REQUEST));

   CtReq->RevisionId.bits.Revision = SLI_CT_REVISION;
   CtReq->RevisionId.bits.InId = 0;

   CtReq->FsType = SLI_CT_DIRECTORY_SERVICE;
   CtReq->FsSubType = SLI_CT_DIRECTORY_NAME_SERVER;

   CtReq->CommandResponse.bits.Size = 0;
   switch (cmdcode) {
   case SLI_CTNS_GID_FT:
      CtReq->CommandResponse.bits.CmdRsp = SWAP_DATA16(SLI_CTNS_GID_FT);
      CtReq->un.gid.Fc4Type = SLI_CTPT_FCP;
      if(binfo->fc_ffstate != FC_READY)
         binfo->fc_ffstate = FC_NS_QRY;
      break;
   case SLI_CTNS_RFT_ID:
      clp = DD_CTL.p_config[binfo->fc_brd_no];
      CtReq->CommandResponse.bits.CmdRsp = SWAP_DATA16(SLI_CTNS_RFT_ID);
      CtReq->un.rft.PortId = SWAP_DATA(binfo->fc_myDID);
      if(clp[CFG_FCP_ON].a_current) {
         CtReq->un.rft.fcpReg = 1;
      }
      if(clp[CFG_NETWORK_ON].a_current) {
         CtReq->un.rft.ipReg = 1;
      }
      if(binfo->fc_ffstate != FC_READY)
         binfo->fc_ffstate = FC_NS_REG;
      break;
   }

   binfo->fc_ns_retry++;
   if(FABRICTMO) {
      fc_clk_can(p_dev_ctl, FABRICTMO);
   }
   FABRICTMO = fc_clk_set(p_dev_ctl, binfo->fc_ratov,
                         fc_fabric_timeout, 0, 0);

   if(fc_ct_cmd(p_dev_ctl, mp, bmp, ndlp)) {
      fc_mem_put(binfo, MEM_BUF, (uchar * )mp);
      fc_mem_put(binfo, MEM_BPL, (uchar * )bmp);
   }
   return(0);
}       /* End fc_ns_cmd */

_static_ int
fc_free_ct_rsp(
fc_dev_ctl_t * p_dev_ctl,
MATCHMAP       * mlist)
{
   FC_BRD_INFO    * binfo;
   MATCHMAP       * mlast;

   binfo = &BINFO;
   while(mlist) {
      mlast = mlist;
      mlist = (MATCHMAP *)mlist->fc_mptr;

      fc_mem_put(binfo, MEM_BUF, (uchar * )mlast);
   }
   return(0);
}

_local_ MATCHMAP *
fc_alloc_ct_rsp(
fc_dev_ctl_t * p_dev_ctl,
ULP_BDE64    * bpl,
uint32         size,
int          * entries)
{
   FC_BRD_INFO     * binfo;
   MATCHMAP        * mlist;
   MATCHMAP        * mlast;
   MATCHMAP        * mp;
   int               cnt, i;

   binfo = &BINFO;
   mlist = 0;
   mlast = 0;
   i = 0;

   while(size) {

      /* We get chucks of FCELSSIZE */
      if(size > FCELSSIZE)
         cnt = FCELSSIZE;
      else
         cnt = size;

      /* Allocate buffer for rsp payload */
      if ((mp = (MATCHMAP * )fc_mem_get(binfo, MEM_BUF)) == 0) {
         fc_free_ct_rsp(p_dev_ctl, mlist);
         return(0);
      }

      /* Queue it to a linked list */
      if(mlast == 0) {
         mlist = mp;
         mlast = mp;
      }
      else {
         mlast->fc_mptr = (uchar *)mp;
         mlast = mp;
      }
      mp->fc_mptr = 0;

      bpl->tus.f.bdeFlags = BUFF_USE_RCV;

      /* build buffer ptr list for IOCB */
      bpl->addrLow = PCIMEM_LONG(putPaddrLow((ulong)mp->phys));
      bpl->addrHigh = PCIMEM_LONG(putPaddrHigh((ulong)mp->phys));
      bpl->tus.f.bdeSize = (ushort)cnt;
      bpl->tus.w = PCIMEM_LONG(bpl->tus.w);
      bpl++;

      i++;
      size -= cnt;
   }

   *entries = i;
   return(mlist);
}

_static_ int
fc_ct_cmd(
fc_dev_ctl_t *p_dev_ctl,
MATCHMAP        *inmp,
MATCHMAP        *bmp,
NODELIST        *ndlp)
{
   FC_BRD_INFO   * binfo;
   ULP_BDE64     * bpl;
   MATCHMAP      * outmp;
   int             cnt;

   binfo = &BINFO;
   bpl = (ULP_BDE64 * )bmp->virt;
   bpl++;  /* Skip past ct request */

   cnt = 0;
   /* Put buffer(s) for ct rsp in bpl */
   if((outmp = fc_alloc_ct_rsp(p_dev_ctl, bpl, FC_MAX_NS_RSP, &cnt)) == 0) {
      return(ENOMEM);
   }

   /* save ndlp for cmpl */
   inmp->fc_mptr = (uchar *)ndlp;

   if((fc_gen_req(binfo, bmp, inmp, outmp, ndlp->nlp_Rpi, 0, (cnt+1), 0))) {
      fc_free_ct_rsp(p_dev_ctl, outmp);
      return(ENOMEM);
   }
   return(0);
}       /* End fc_ct_cmd */


/**************************************************/
/**  fc_ns_rsp                                   **/
/**                                              **/
/**  Description:                                **/
/**  Process NameServer response                 **/
/**                                              **/
/**    Returns:                                  **/
/**                                              **/
/**************************************************/
_static_ int
fc_ns_rsp(
fc_dev_ctl_t *p_dev_ctl,
NODELIST     *nslp,
MATCHMAP     *mp,
uint32        Size)
{
   FC_BRD_INFO    * binfo;
   SLI_CT_REQUEST * Response;
   NODELIST       * ndlp;
   NODELIST       * new_ndlp;
   MATCHMAP       * mlast;
   D_ID           rscn_did;
   D_ID           ns_did;
   uint32         * tptr;
   uint32         Did;
   uint32         Temp;
   int            j, Cnt, match, new_node;

   binfo = &BINFO;
   ndlp = 0;
   binfo->fc_ns_retry = 0;

   binfo->fc_fabrictmo = (2 * binfo->fc_ratov) +
      ((4 * binfo->fc_edtov) / 1000) + 1;
   if(FABRICTMO) {
      fc_clk_can(p_dev_ctl, FABRICTMO);
      FABRICTMO = 0;
   }

   Response = (SLI_CT_REQUEST * )mp->virt;

   if ((Response->CommandResponse.bits.CmdRsp == SWAP_DATA16(SLI_CT_RESPONSE_FS_ACC)) && 
       ((binfo->fc_ffstate == FC_NS_QRY) ||
       ((binfo->fc_ffstate == FC_READY) && (binfo->fc_flag & FC_RSCN_MODE)))) {

      tptr = (uint32 * ) & Response->un.gid.PortType;
      while(mp) {
         mlast = mp;
         mp = (MATCHMAP *)mp->fc_mptr;
         fc_mpdata_sync(mlast->dma_handle, 0, 0, DDI_DMA_SYNC_FORKERNEL);

         if(Size > FCELSSIZE)
            Cnt = FCELSSIZE;
         else
            Cnt = Size;
         Size -= Cnt;

         if(tptr == 0)
            tptr = (uint32 * )mlast->virt;
         else
            Cnt -= 16;  /* subtract length of CT header */

         while(Cnt) {
         /* Loop through entire NameServer list of DIDs */

            /* Get next DID from NameServer List */
            Temp = *tptr++;
            Did = (SWAP_DATA(Temp) & Mask_DID);

            ndlp = 0;
            if ((Did) && (Did != binfo->fc_myDID)) {
               new_node = 0;
               ndlp = fc_findnode_odid(binfo, NLP_SEARCH_ALL, Did);
               if(ndlp) {
                  ndlp->nlp_DID = Did;
                  /* Skip nodes already marked for discovery / rscn */
                  if(ndlp->nlp_action &
                     (NLP_DO_ADDR_AUTH | NLP_DO_DISC_START | NLP_DO_RSCN))
                     goto nsout;
               }
               else {
                  new_node = 1;
                  if((ndlp = (NODELIST *)fc_mem_get(binfo, MEM_NLP))) {
                     fc_bzero((void *)ndlp, sizeof(NODELIST));
                     ndlp->sync = binfo->fc_sync;
                     ndlp->capabilities = binfo->fc_capabilities;
                     ndlp->nlp_DID = Did;
                     fc_nlp_bind(binfo, ndlp);
                  }
                  else
                     goto nsout;
               }
      
               if ((new_node) || 
                   (!(ndlp->nlp_flag & NLP_REQ_SND) &&
                     (ndlp->nlp_state < NLP_ALLOC)) ) {

                  if ((binfo->fc_ffstate == FC_READY) &&
                     (binfo->fc_flag & FC_RSCN_MODE)) {
                     /* we are in RSCN node, so match Did from NameServer with
                      * with list recieved from previous RSCN commands.
                      * Do NOT add it to our RSCN discovery list unless we have
                      * a match.
                      */
                     match = 0;
                     for(j=0;j<binfo->fc_rscn_id_cnt;j++) {

                        rscn_did.un.word = binfo->fc_rscn_id_list[j];
                        ns_did.un.word = Did;

                        switch (rscn_did.un.b.resv) {
                        case 0:   /* Single N_Port ID effected */
                           if (ns_did.un.word == rscn_did.un.word) {
                              match = 1;
                           }
                           break;

                        case 1:   /* Whole N_Port Area effected */
                           if ((ns_did.un.b.domain == rscn_did.un.b.domain) && 
                               (ns_did.un.b.area   == rscn_did.un.b.area)) {
                              match = 1;
                           }
                           break;

                        case 2:   /* Whole N_Port Domain effected */
                           if (ns_did.un.b.domain == rscn_did.un.b.domain) {
                              match = 1;
                           }
                           break;

                        case 3:   /* Whole Fabric effected */
                           match = 1;
                           break;

                        default:
                           /* Unknown Identifier in RSCN list */
                           fc_log_printf_msg_vargs( binfo->fc_brd_no,
                                  &fc_msgBlk0237,                   /* ptr to msg structure */
                                   fc_mes0237,                      /* ptr to msg */
                                    fc_msgBlk0237.msgPreambleStr,   /* begin varargs */
                                     rscn_did.un.word);             /* end varargs */
                           break;

                        }
                        if(match)
                           break;
                     }
                     if(match == 0)  /* Skip it */
                        goto nsout;
                  }

                  /* Add it to our discovery list */
                  ndlp->nlp_state = NLP_LIMBO;
                  fc_nlp_bind(binfo, ndlp);
                  if ((binfo->fc_ffstate == FC_READY) &&
                     (binfo->fc_flag & FC_RSCN_MODE)) {
                     ndlp->nlp_action |= NLP_DO_RSCN;
                     ndlp->nlp_flag &= ~NLP_NODEV_TMO;
                  }
                  else {
                     ndlp->nlp_action |= NLP_DO_DISC_START;
                  }
               }
               else {
                  if (binfo->fc_ffstate < FC_READY) {
                     /* Add it to our discovery list */
                     ndlp->nlp_state = NLP_LIMBO;
                     fc_nlp_bind(binfo, ndlp);
                     ndlp->nlp_action |= NLP_DO_DISC_START;
                  }
               }
            }
nsout:

            /* Mark all node table entries that are in the Nameserver */
            if(ndlp) {
               ndlp->nlp_flag |= NLP_NS_NODE;
               ndlp->nlp_flag &= ~NLP_NS_REMOVED;
               /* NameServer Rsp */
               fc_log_printf_msg_vargs( binfo->fc_brd_no,
                      &fc_msgBlk0238,                   /* ptr to msg structure */
                       fc_mes0238,                      /* ptr to msg */
                        fc_msgBlk0238.msgPreambleStr,   /* begin varargs */
                         Did,
                          ndlp->nlp_flag,
                           binfo->fc_flag,
                            binfo->fc_rscn_id_cnt);     /* end varargs */
            }
            else {
               /* NameServer Rsp */
               fc_log_printf_msg_vargs( binfo->fc_brd_no,
                      &fc_msgBlk0239,                   /* ptr to msg structure */
                       fc_mes0239,                      /* ptr to msg */
                        fc_msgBlk0239.msgPreambleStr,   /* begin varargs */
                         Did,
                          (ulong)ndlp,
                            binfo->fc_flag,
                             binfo->fc_rscn_id_cnt);    /* end varargs */
            }

            if (Temp & SWAP_DATA(SLI_CT_LAST_ENTRY))
               goto nsout1;
            Cnt -= sizeof(uint32);
         }
         tptr = 0;
      }

nsout1:
      /* Take out all node table entries that are not in the NameServer */
      ndlp = binfo->fc_nlpbind_start;
      if(ndlp == (NODELIST *)&binfo->fc_nlpbind_start)
        ndlp = binfo->fc_nlpunmap_start;
      if(ndlp == (NODELIST *)&binfo->fc_nlpunmap_start)
         ndlp = binfo->fc_nlpmap_start;
      while(ndlp != (NODELIST *)&binfo->fc_nlpmap_start) {
         new_ndlp = (NODELIST *)ndlp->nlp_listp_next;
         if ( (ndlp->nlp_state == NLP_LIMBO) ||
             (ndlp->nlp_state == NLP_SEED) ||
             (ndlp->nlp_DID == binfo->fc_myDID) || 
             (ndlp->nlp_DID == NameServer_DID) || 
             (ndlp->nlp_DID == FDMI_DID) || 
             (ndlp->nlp_type & NLP_FABRIC) || 
             (ndlp->nlp_flag & NLP_NS_NODE)) {
            if(ndlp->nlp_flag & NLP_NS_NODE) {
               ndlp->nlp_flag &= ~NLP_NS_NODE;
            } else {
               if(ndlp->nlp_DID != NameServer_DID)
                  ndlp->nlp_action &= ~(NLP_DO_ADDR_AUTH | NLP_DO_DISC_START | NLP_DO_RSCN);
            }
            goto loop1;
         }
         if ((binfo->fc_ffstate == FC_READY) &&
             (binfo->fc_flag & FC_RSCN_MODE) &&
             !(ndlp->nlp_action & NLP_DO_RSCN))
            goto loop1;

         if ((ndlp->nlp_DID != 0) && !(ndlp->nlp_flag & NLP_NODEV_TMO)) {
            RING       * rp;
            IOCBQ      * iocbq;
            /* Look through ELS ring and remove any ELS cmds in progress */
            rp = &binfo->fc_ring[FC_ELS_RING];
            iocbq = (IOCBQ * )(rp->fc_txp.q_first);
            while (iocbq) {
               if(iocbq->iocb.un.elsreq.remoteID == ndlp->nlp_DID) {
                  iocbq->retry = 0xff;  /* Mark for abort */
               }
               iocbq = (IOCBQ * )iocbq->q;
            }
            /* In case its on fc_delay_timeout list */
            fc_abort_delay_els_cmd(p_dev_ctl, ndlp->nlp_DID);

            ndlp->nlp_flag &= ~(NLP_REQ_SND | NLP_REQ_SND_ADISC);
         }

         ndlp->nlp_action &= ~(NLP_DO_ADDR_AUTH | NLP_DO_DISC_START | NLP_DO_RSCN);
         fc_freenode(binfo, ndlp, 0);
         ndlp->nlp_state = NLP_LIMBO;
         fc_nlp_bind(binfo, ndlp);
         ndlp->nlp_flag |= NLP_NS_REMOVED;
         ndlp->nlp_type &= ~(NLP_FABRIC | NLP_IP_NODE); 
loop1:
         ndlp = new_ndlp;
         if(ndlp == (NODELIST *)&binfo->fc_nlpbind_start)
           ndlp = binfo->fc_nlpunmap_start;
         if(ndlp == (NODELIST *)&binfo->fc_nlpunmap_start)
            ndlp = binfo->fc_nlpmap_start;
      }

   } else if ((Response->CommandResponse.bits.CmdRsp == SWAP_DATA16(SLI_CT_RESPONSE_FS_RJT)) && 
              ((binfo->fc_ffstate == FC_NS_QRY) ||
               ((binfo->fc_ffstate == FC_READY) && (binfo->fc_flag & FC_RSCN_MODE)))) {
       /* NameServer Rsp Error */
       fc_log_printf_msg_vargs( binfo->fc_brd_no,
              &fc_msgBlk0240,                   /* ptr to msg structure */
               fc_mes0240,                      /* ptr to msg */
                fc_msgBlk0240.msgPreambleStr,   /* begin varargs */
                 Response->CommandResponse.bits.CmdRsp,
                  (uint32)Response->ReasonCode,
                   (uint32)Response->Explanation,
                    binfo->fc_flag);            /* end varargs */
      goto nsout1;

   } else {
       /* NameServer Rsp Error */
       fc_log_printf_msg_vargs( binfo->fc_brd_no,
              &fc_msgBlk0241,                   /* ptr to msg structure */
               fc_mes0241,                      /* ptr to msg */
                fc_msgBlk0241.msgPreambleStr,   /* begin varargs */
                 Response->CommandResponse.bits.CmdRsp,
                  (uint32)Response->ReasonCode,
                   (uint32)Response->Explanation,
                    binfo->fc_flag);            /* end varargs */
   }

   if (binfo->fc_ffstate == FC_NS_REG) {
      /* Issue GID_FT to Nameserver */
      if (fc_ns_cmd(p_dev_ctl, nslp, SLI_CTNS_GID_FT))
         goto out;
   } else {
out:
      /* Done with NameServer for now, but leave logged in */

      /* We can start discovery right now */
      /* Fire out PLOGIs on all nodes marked for discovery */
      binfo->fc_rscn_id_cnt = 0;
      if ((binfo->fc_nlp_cnt <= 0) && !(binfo->fc_flag & FC_NLP_MORE)) {
         binfo->fc_nlp_cnt = 0;
         if ((binfo->fc_ffstate == FC_READY) &&
             (binfo->fc_flag & FC_RSCN_MODE)) {
            nslp->nlp_action &= ~(NLP_DO_ADDR_AUTH | NLP_DO_RSCN);
            fc_nextrscn(p_dev_ctl, fc_max_els_sent);
         }
         else {
            nslp->nlp_action |= NLP_DO_ADDR_AUTH;
            fc_nextnode(p_dev_ctl, nslp);
         }
      }
      else {
         nslp->nlp_action |= NLP_DO_ADDR_AUTH;
         fc_nextnode(p_dev_ctl, nslp);
      }
   }
   return(0);
}       /* End fc_ns_rsp */

/**************************************************/
/**  fc_free_clearq                              **/
/**                                              **/
/**  Description:                                **/
/**  Called to free all clearq bufs for a device **/
/**                                              **/
/**    Returns:                                  **/
/**                                              **/
/**************************************************/
_static_ void
fc_free_clearq(
dvi_t        *dev_ptr)
{
   struct buf *bp, *nextbp;
   FC_BRD_INFO  * binfo;

   binfo = &dev_ptr->nodep->ap->info;

   /* Call iodone for all the CLEARQ error bufs */
   for (bp = dev_ptr->clear_head; bp != NULL; ) {
      dev_ptr->clear_count--;
      nextbp = bp->av_forw;
      FCSTATCTR.fcpScsiTmo++;
      fc_do_iodone(bp);
      bp = nextbp;
   }
   dev_ptr->clear_head = NULL;
   dev_ptr->flags &= ~SCSI_TQ_HALTED & ~SCSI_TQ_CLEARING;

   fc_restart_device(dev_ptr);
   return;
}       /* End fc_free_clearq */


/****************************************************/
/**  fc_nextnode                                   **/
/**                                                **/
/**  Description:                                  **/
/**  Called during discovery or rediscovery        **/
/**                                                **/
/**    Returns:                                    **/
/**                                                **/
/****************************************************/
_static_ int
fc_nextnode(
fc_dev_ctl_t *p_dev_ctl,
NODELIST     *ndlp)
{
   FC_BRD_INFO   * binfo;
   node_t        * node_ptr;
   dvi_t         * dev_ptr;
   iCfgParam     * clp;

   binfo = &BINFO;
   binfo->fc_fabrictmo = (2 * binfo->fc_ratov) +
      ((4 * binfo->fc_edtov) / 1000) + 1;

   clp = DD_CTL.p_config[binfo->fc_brd_no];

   /* Device Discovery nextnode */
   fc_log_printf_msg_vargs( binfo->fc_brd_no,
          &fc_msgBlk0242,                   /* ptr to msg structure */
           fc_mes0242,                      /* ptr to msg */
            fc_msgBlk0242.msgPreambleStr,   /* begin varargs */
             (uint32)ndlp->nlp_state,
              ndlp->nlp_DID,
               (uint32)ndlp->nlp_flag,
                binfo->fc_ffstate);         /* end varargs */
   if (binfo->fc_flag & FC_FABRIC) {
      if (binfo->fc_ffstate < FC_NS_QRY) {
         return(0);
      }
      if ((binfo->fc_ffstate < FC_NODE_DISC) && binfo->fc_ns_retry) {
         return(0);
      }
   }

   if(FABRICTMO) {
      fc_clk_res(p_dev_ctl, binfo->fc_fabrictmo, FABRICTMO);
   }
   else {
      FABRICTMO = fc_clk_set(p_dev_ctl, binfo->fc_fabrictmo,
        fc_fabric_timeout, 0, 0);
   }

   if ((ndlp->nlp_type & NLP_FCP_TARGET) && (ndlp->nlp_state == NLP_ALLOC)) {
      if(clp[CFG_FIRST_CHECK].a_current) {
         /* If we are an FCP node, update first_check flag for all LUNs */
         if ((node_ptr = (node_t * )ndlp->nlp_targetp) != NULL) {
            for (dev_ptr = node_ptr->lunlist; dev_ptr != NULL; 
                dev_ptr = dev_ptr->next) {
               dev_ptr->first_check = FIRST_CHECK_COND;
               fc_device_changed(p_dev_ctl, dev_ptr);
            }
         }
      }
   }

   /* Check for ADISC Address Authentication */
   if (ndlp->nlp_action & NLP_DO_ADDR_AUTH) {
      ndlp->nlp_flag &= ~(NLP_REQ_SND | NLP_REQ_SND_ADISC);
      ndlp->nlp_action &= ~NLP_DO_ADDR_AUTH;

      if(ndlp->nlp_DID != NameServer_DID)
         binfo->fc_nlp_cnt--;

      if (binfo->fc_nlp_cnt <= 0) {
         /* If no nodes left to authenticate, redo discovery on any
          * new nodes.
          */
         if (fc_nextauth(p_dev_ctl, fc_max_els_sent) == 0) {
            binfo->fc_nlp_cnt = 0;
            fc_nextdisc(p_dev_ctl, fc_max_els_sent);
         }
      } else {
         fc_nextauth(p_dev_ctl, 1);
      }

      return(0);
   }

   /* Check for RSCN Discovery */
   if (ndlp->nlp_action & NLP_DO_RSCN) {
      ndlp->nlp_flag &= ~(NLP_REQ_SND | NLP_REQ_SND_ADISC);
      ndlp->nlp_action &= ~NLP_DO_RSCN;
      binfo->fc_nlp_cnt--;
      if ((ndlp->nlp_type & NLP_IP_NODE) && ndlp->nlp_bp) {
         m_freem((fcipbuf_t *)ndlp->nlp_bp);
         ndlp->nlp_bp = (uchar * )0;
      }

      if (ndlp->nlp_type & NLP_FCP_TARGET) {
         node_t  * node_ptr;
         dvi_t   * dev_ptr;

         if ((node_ptr = (node_t * )ndlp->nlp_targetp) != NULL) {
            /* restart any I/Os on this node */
            for (dev_ptr = node_ptr->lunlist;
               dev_ptr != NULL; dev_ptr = dev_ptr->next) {
               dev_ptr->queue_state = HALTED;
            }
         }
      }

      if (binfo->fc_nlp_cnt <= 0) {
         binfo->fc_nlp_cnt = 0;
         fc_nextrscn(p_dev_ctl, fc_max_els_sent);
      } else {
         fc_nextrscn(p_dev_ctl, 1);
      }
   }

   /* Check for Address Discovery */
   if ((ndlp->nlp_action & NLP_DO_DISC_START) ||
      (ndlp->nlp_flag & NLP_REQ_SND)) {
      ndlp->nlp_flag &= ~NLP_REQ_SND;
      ndlp->nlp_action &= ~NLP_DO_DISC_START;
      binfo->fc_nlp_cnt--;

      if (binfo->fc_nlp_cnt <= 0) {
         binfo->fc_nlp_cnt = 0;
         fc_nextdisc(p_dev_ctl, fc_max_els_sent);
      } else {
         fc_nextdisc(p_dev_ctl, 1);
      }
   }

   return(0);
}       /* End fc_nextnode */


/****************************************************/
/**  fc_nextdisc                                   **/
/**                                                **/
/**  Description:                                  **/
/**  Called during discovery or rediscovery        **/
/**                                                **/
/**    Returns:                                    **/
/**                                                **/
/****************************************************/
_static_ int
fc_nextdisc(
fc_dev_ctl_t *p_dev_ctl,
int sndcnt)
{
   FC_BRD_INFO   * binfo;
   MAILBOXQ      * mb;
   NODELIST      * ndlp;
   NODELIST      * new_ndlp;
   int             cnt, skip;
   uint32          did;

   binfo = &BINFO;
   /* Device Discovery nextdisc */
   fc_log_printf_msg_vargs( binfo->fc_brd_no,
          &fc_msgBlk0243,                   /* ptr to msg structure */
           fc_mes0243,                      /* ptr to msg */
            fc_msgBlk0243.msgPreambleStr,   /* begin varargs */
             binfo->fc_nlp_cnt,
              sndcnt,
               binfo->fc_mbox_active);      /* end varargs */
   binfo->fc_ffstate = FC_NODE_DISC;
   binfo->fc_fabrictmo = (2 * binfo->fc_ratov) +
      ((4 * binfo->fc_edtov) / 1000) + 1;
   if(FABRICTMO) {
      fc_clk_res(p_dev_ctl, binfo->fc_fabrictmo, FABRICTMO);
   }
   else {
      FABRICTMO = fc_clk_set(p_dev_ctl, binfo->fc_fabrictmo,
                         fc_fabric_timeout, 0, 0);
   }

   /* For MAXREQ requests, we must make sure all outstanding Mailbox
    * commands have been processed. This is to ensure UNREG_LOGINs
    * complete before we try to relogin.
    */
   if (sndcnt == fc_max_els_sent) {
      if (binfo->fc_mbox_active) {
         binfo->fc_flag |= FC_DELAY_PLOGI;
         return(fc_max_els_sent);
      }
   }

   cnt = 0;
   skip = 0;
   binfo->fc_flag &= ~FC_NLP_MORE;

   /* We can start discovery right now */
   /* Fire out PLOGIs on all nodes marked for discovery */
   ndlp = binfo->fc_nlpbind_start;
   if(ndlp == (NODELIST *)&binfo->fc_nlpbind_start)
     ndlp = binfo->fc_nlpunmap_start;
   if(ndlp == (NODELIST *)&binfo->fc_nlpunmap_start)
      ndlp = binfo->fc_nlpmap_start;
   while(ndlp != (NODELIST *)&binfo->fc_nlpmap_start) {
      new_ndlp = (NODELIST *)ndlp->nlp_listp_next;

      if ((ndlp->nlp_action & NLP_DO_DISC_START) && 
          (ndlp->nlp_DID != NameServer_DID)) {
         if(!(ndlp->nlp_flag & (NLP_REQ_SND | NLP_REG_INP | NLP_RM_ENTRY))) {
            binfo->fc_nlp_cnt++;
            did = ndlp->nlp_DID;
            if(did == 0)
               did = ndlp->nlp_oldDID;
            /* Start discovery for this node */
            fc_els_cmd(binfo, ELS_CMD_PLOGI, (void *)((ulong)did),
                (uint32)0, (ushort)0, ndlp);
            cnt++;

            if ((binfo->fc_nlp_cnt >= fc_max_els_sent) || 
                (cnt == sndcnt)) {
               binfo->fc_flag |= FC_NLP_MORE;
               return(cnt);
            }
         }
         else
            skip++;
      }
      ndlp = new_ndlp;
      if(ndlp == (NODELIST *)&binfo->fc_nlpbind_start)
        ndlp = binfo->fc_nlpunmap_start;
      if(ndlp == (NODELIST *)&binfo->fc_nlpunmap_start)
         ndlp = binfo->fc_nlpmap_start;
   }

   if ((binfo->fc_nlp_cnt) || skip)
      return(binfo->fc_nlp_cnt);

   /* This should turn off DELAYED ABTS for ELS timeouts */
   if ((mb = (MAILBOXQ * )fc_mem_get(binfo, MEM_MBOX))) {
      fc_set_slim(binfo, (MAILBOX * )mb, 0x052198, 0);
      if (issue_mb_cmd(binfo, (MAILBOX * )mb, MBX_NOWAIT) != MBX_BUSY) {
         fc_mem_put(binfo, MEM_MBOX, (uchar * )mb);
      }
   }

   /* Nothing to authenticate, so CLEAR_LA right now */
   if ((mb = (MAILBOXQ * )fc_mem_get(binfo, MEM_MBOX | MEM_PRI))) {
      binfo->fc_ffstate = FC_CLEAR_LA;
      fc_clear_la(binfo, (MAILBOX * )mb);
      if (issue_mb_cmd(binfo, (MAILBOX * )mb, MBX_NOWAIT) != MBX_BUSY) {
         fc_mem_put(binfo, MEM_MBOX, (uchar * )mb);
      }
   } else {
      /* Device Discovery completion error */
      fc_log_printf_msg_vargs( binfo->fc_brd_no,
             &fc_msgBlk0244,                   /* ptr to msg structure */
              fc_mes0244,                      /* ptr to msg */
               fc_msgBlk0244.msgPreambleStr);  /* begin & end varargs */
      binfo->fc_ffstate = FC_ERROR;
   }

   if(FABRICTMO) {
      fc_clk_can(p_dev_ctl, FABRICTMO);
      FABRICTMO = 0;
   }
   return(0);
}       /* End fc_nextdisc */


/****************************************************/
/**  fc_nextauth                                   **/
/**                                                **/
/**  Description:                                  **/
/**  Called during rediscovery                     **/
/**                                                **/
/**    Returns:                                    **/
/**                                                **/
/****************************************************/
_static_ int
fc_nextauth(
fc_dev_ctl_t *p_dev_ctl,
int sndcnt)
{
   FC_BRD_INFO   * binfo;
   iCfgParam     * clp;
   NODELIST      * ndlp;
   NODELIST      * new_ndlp;
   int             cnt;

   binfo = &BINFO;
   clp = DD_CTL.p_config[binfo->fc_brd_no];
   /* Device Discovery next authentication */
   fc_log_printf_msg_vargs( binfo->fc_brd_no,
          &fc_msgBlk0245,                   /* ptr to msg structure */
           fc_mes0245,                      /* ptr to msg */
            fc_msgBlk0245.msgPreambleStr,   /* begin varargs */
             binfo->fc_nlp_cnt,
              sndcnt,
               binfo->fc_mbox_active);      /* end varargs */
   cnt = 0;
   binfo->fc_flag &= ~FC_NLP_MORE;
   binfo->fc_fabrictmo = (2 * binfo->fc_ratov) +
      ((4 * binfo->fc_edtov) / 1000) + 1;
   if(FABRICTMO) {
      fc_clk_res(p_dev_ctl, binfo->fc_fabrictmo, FABRICTMO);
   }
   else {
      FABRICTMO = fc_clk_set(p_dev_ctl, binfo->fc_fabrictmo,
                         fc_fabric_timeout, 0, 0);
   }

   /* We can start rediscovery right now */
   /* Fire out ADISC on all nodes marked for addr_auth */

   ndlp = binfo->fc_nlpbind_start;
   if(ndlp == (NODELIST *)&binfo->fc_nlpbind_start)
     ndlp = binfo->fc_nlpunmap_start;
   if(ndlp == (NODELIST *)&binfo->fc_nlpunmap_start)
      ndlp = binfo->fc_nlpmap_start;
   while(ndlp != (NODELIST *)&binfo->fc_nlpmap_start) {
      new_ndlp = (NODELIST *)ndlp->nlp_listp_next;

      if (ndlp->nlp_action & NLP_DO_ADDR_AUTH) {
         if (ndlp->nlp_flag & (NLP_RM_ENTRY | NLP_REQ_SND_ADISC |
            NLP_REQ_SND | NLP_REG_INP))
            goto loop1;

         if ((ndlp->nlp_type & NLP_FCP_TARGET) && 
             ((!clp[CFG_USE_ADISC].a_current) || (ndlp->nlp_Rpi == 0))) {
            /* Force regular discovery on this node */
            ndlp->nlp_action &= ~NLP_DO_ADDR_AUTH;
            fc_freenode(binfo, ndlp, 0); 
            ndlp->nlp_state = NLP_LIMBO;
            fc_nlp_bind(binfo, ndlp);
            ndlp->nlp_action |= NLP_DO_DISC_START;
            goto loop1;
         } else {
            if ((ndlp->nlp_type & NLP_IP_NODE) && (ndlp->nlp_Rpi == 0)) {
               /* Force regular discovery on this node */
               ndlp->nlp_action &= ~NLP_DO_ADDR_AUTH;
               fc_freenode(binfo, ndlp, 0);
               ndlp->nlp_state = NLP_LIMBO;
               fc_nlp_bind(binfo, ndlp);
               goto loop1;
            }
         }

         if((ndlp->nlp_type & (NLP_FCP_TARGET | NLP_IP_NODE)) == 0) {
            /* Force regular discovery on this node */
            ndlp->nlp_action &= ~NLP_DO_ADDR_AUTH;
            fc_freenode(binfo, ndlp, 0);
            ndlp->nlp_state = NLP_LIMBO;
            fc_nlp_bind(binfo, ndlp);
            ndlp->nlp_action |= NLP_DO_DISC_START;
            goto loop1;
         }

         binfo->fc_nlp_cnt++;
         /* Start authentication */
         fc_els_cmd(binfo, ELS_CMD_ADISC, (void *)((ulong)ndlp->nlp_DID),
             (uint32)0, (ushort)0, ndlp);
         cnt++;
         if ((binfo->fc_nlp_cnt >= fc_max_els_sent) || 
             (cnt == sndcnt)) {
            binfo->fc_flag |= FC_NLP_MORE;
            return(cnt);
         }
      }
loop1:
      ndlp = new_ndlp;
      if(ndlp == (NODELIST *)&binfo->fc_nlpbind_start)
        ndlp = binfo->fc_nlpunmap_start;
      if(ndlp == (NODELIST *)&binfo->fc_nlpunmap_start)
         ndlp = binfo->fc_nlpmap_start;
   }

   return(binfo->fc_nlp_cnt);
}       /* End fc_nextauth */

/****************************************************/
/**  fc_nextrscn                                   **/
/**                                                **/
/**  Description:                                  **/
/**  Called during RSCN processing                 **/
/**                                                **/
/**    Returns:                                    **/
/**                                                **/
/****************************************************/
_static_ int
fc_nextrscn(
fc_dev_ctl_t *p_dev_ctl,
int sndcnt)
{
   FC_BRD_INFO   * binfo;
   iCfgParam     * clp;
   NODELIST      * ndlp;
   NODELIST      * new_ndlp;
   MAILBOXQ      * mb;
   struct buf    * bp, * nextbp;
   RING          * rp;
   IOCBQ         * xmitiq;
   IOCB          * iocb;
   MATCHMAP      * mp;
   int  i, j, cnt;
   uint32  did;

   binfo = &BINFO;
   clp = DD_CTL.p_config[binfo->fc_brd_no];
   /* Device Discovery next RSCN */
   fc_log_printf_msg_vargs( binfo->fc_brd_no,
          &fc_msgBlk0246,                   /* ptr to msg structure */
           fc_mes0246,                      /* ptr to msg */
            fc_msgBlk0246.msgPreambleStr,   /* begin varargs */
             binfo->fc_nlp_cnt,
              sndcnt,
               binfo->fc_mbox_active,
                binfo->fc_flag);            /* end varargs */
   cnt = 0;
   if (binfo->fc_flag & FC_RSCN_DISC_TMR)
      goto out;

   /* Are we waiting for a NameServer Query to complete */
   if (binfo->fc_flag & FC_NSLOGI_TMR)
      return(1);

   if (binfo->fc_mbox_active) {
      binfo->fc_flag |= FC_DELAY_PLOGI;
      return(1);
   }

   binfo->fc_flag &= ~FC_NLP_MORE;

   if(FABRICTMO) {
      fc_clk_res(p_dev_ctl, binfo->fc_fabrictmo, FABRICTMO);
   }
   else {
      FABRICTMO = fc_clk_set(p_dev_ctl, binfo->fc_fabrictmo,
                         fc_fabric_timeout, 0, 0);
   }

   /* We can start rediscovery right now */
   /* Fire out PLOGI on all nodes marked for rscn */
   ndlp = binfo->fc_nlpbind_start;
   if(ndlp == (NODELIST *)&binfo->fc_nlpbind_start)
     ndlp = binfo->fc_nlpunmap_start;
   if(ndlp == (NODELIST *)&binfo->fc_nlpunmap_start)
      ndlp = binfo->fc_nlpmap_start;
   while(ndlp != (NODELIST *)&binfo->fc_nlpmap_start) {
      new_ndlp = (NODELIST *)ndlp->nlp_listp_next;

      if (ndlp->nlp_action & NLP_DO_RSCN) {
         if (ndlp->nlp_flag & (NLP_RM_ENTRY | NLP_REQ_SND_ADISC |
            NLP_REQ_SND | NLP_REG_INP))
            goto loop2;

         did = ndlp->nlp_DID;
         if(did == 0) {
            did = ndlp->nlp_oldDID;
            if(did == 0)
               goto loop2;
         }

         binfo->fc_nlp_cnt++;

         /* We are always using ADISC for RSCN validation */
         if ((ndlp->nlp_type & NLP_FCP_TARGET) && (ndlp->nlp_Rpi == 0)) {
            /* Force regular discovery on this node */
            fc_els_cmd(binfo, ELS_CMD_PLOGI, (void *)((ulong)did),
                (uint32)0, (ushort)0, ndlp);
         } else {
            if (((ndlp->nlp_type & NLP_IP_NODE) && (ndlp->nlp_Rpi == 0)) ||
               ((ndlp->nlp_type & (NLP_FCP_TARGET | NLP_IP_NODE)) == 0)) {
               /* Force regular discovery on this node */
               fc_els_cmd(binfo, ELS_CMD_PLOGI, (void *)((ulong)did),
                   (uint32)0, (ushort)0, ndlp);
            }
            else {
               fc_els_cmd(binfo, ELS_CMD_ADISC,(void *)((ulong)did),
                (uint32)0, (ushort)0, ndlp);
            }
         }
         cnt++;
         if ((binfo->fc_nlp_cnt >= fc_max_els_sent) || 
             (cnt == sndcnt)) {
            binfo->fc_flag |= FC_NLP_MORE;
            return(cnt);
         }
      }
loop2:
      ndlp = new_ndlp;
      if(ndlp == (NODELIST *)&binfo->fc_nlpbind_start)
        ndlp = binfo->fc_nlpunmap_start;
      if(ndlp == (NODELIST *)&binfo->fc_nlpunmap_start)
         ndlp = binfo->fc_nlpmap_start;
   }

   if (binfo->fc_nlp_cnt)
      return(binfo->fc_nlp_cnt);

out:
   if (binfo->fc_flag & FC_RSCN_DISCOVERY) {
      /* Discovery RSCN */
      fc_log_printf_msg_vargs( binfo->fc_brd_no,
             &fc_msgBlk0247,                   /* ptr to msg structure */
              fc_mes0247,                      /* ptr to msg */
               fc_msgBlk0247.msgPreambleStr,   /* begin varargs */
                binfo->fc_defer_rscn.q_cnt,
                 binfo->fc_flag,
                  (ulong)(binfo->fc_rscn_disc_wdt)); /* end varargs */
      if(binfo->fc_rscn_disc_wdt == 0) {
         binfo->fc_rscn_disc_wdt = fc_clk_set(p_dev_ctl,
            ((binfo->fc_edtov / 1000) + 1), fc_rscndisc_timeout, 0, 0);
         /* Free any deferred RSCNs */
         fc_flush_rscn_defer(p_dev_ctl);
         return(0);
      }

      if(!(binfo->fc_flag & FC_RSCN_DISC_TMR))
         return(0);

      binfo->fc_flag &= ~(FC_RSCN_DISC_TMR | FC_RSCN_DISCOVERY);
      binfo->fc_rscn_disc_wdt = 0;

      /* RSCN match on all DIDs in NameServer */
      binfo->fc_rscn_id_list[0] = 0x03000000;
      binfo->fc_rscn_id_cnt = 1;

      /* Free any deferred RSCNs */
      fc_flush_rscn_defer(p_dev_ctl);

      /* Authenticate all nodes in nlplist */
      ndlp = binfo->fc_nlpbind_start;
      if(ndlp == (NODELIST *)&binfo->fc_nlpbind_start)
        ndlp = binfo->fc_nlpunmap_start;
      if(ndlp == (NODELIST *)&binfo->fc_nlpunmap_start)
         ndlp = binfo->fc_nlpmap_start;
      while(ndlp != (NODELIST *)&binfo->fc_nlpmap_start) {
         new_ndlp = (NODELIST *)ndlp->nlp_listp_next;

         /* Skip over FABRIC nodes and myself */
         if ((ndlp->nlp_DID == binfo->fc_myDID) || 
             (ndlp->nlp_type & NLP_FABRIC) || 
             ((ndlp->nlp_DID & CT_DID_MASK) == CT_DID_MASK))
            goto loop3;

         if (ndlp->nlp_state == NLP_ALLOC) {
            /* Mark node for authentication */
            ndlp->nlp_action |= NLP_DO_RSCN;
            ndlp->nlp_flag &= ~NLP_NODEV_TMO;

            /* We are always using ADISC for RSCN validation */
         }
loop3:
         ndlp = new_ndlp;
         if(ndlp == (NODELIST *)&binfo->fc_nlpbind_start)
           ndlp = binfo->fc_nlpunmap_start;
         if(ndlp == (NODELIST *)&binfo->fc_nlpunmap_start)
            ndlp = binfo->fc_nlpmap_start;
      }

      fc_issue_ns_query(p_dev_ctl, (void *)0, (void *)0);
      return(0);
   }

   if (binfo->fc_defer_rscn.q_first) {
      uint32       * lp;
      D_ID         rdid;
      uint32       cmd;

      /* process any deferred RSCNs */
      /* Deferred RSCN */
      fc_log_printf_msg_vargs( binfo->fc_brd_no,
             &fc_msgBlk0248,                   /* ptr to msg structure */
              fc_mes0248,                      /* ptr to msg */
               fc_msgBlk0248.msgPreambleStr,   /* begin varargs */
                binfo->fc_defer_rscn.q_cnt,
                 binfo->fc_flag);              /* end varargs */
      binfo->fc_rscn_id_cnt = 0;
      rp = &binfo->fc_ring[FC_ELS_RING];
      while (binfo->fc_defer_rscn.q_first) {
         xmitiq = (IOCBQ * )binfo->fc_defer_rscn.q_first;
         if ((binfo->fc_defer_rscn.q_first = xmitiq->q) == 0) {
            binfo->fc_defer_rscn.q_last = 0;
         }
         binfo->fc_defer_rscn.q_cnt--;
         iocb = &xmitiq->iocb;
         mp = *((MATCHMAP **)iocb);
         *((MATCHMAP **)iocb) = 0;
         xmitiq->q = NULL;

         lp = (uint32 * )mp->virt;
         cmd = *lp++;
         i = SWAP_DATA(cmd) & 0xffff;   /* payload length */
         i -= sizeof(uint32);  /* take off word 0 */
         while (i) {
            rdid.un.word = *lp++;
            rdid.un.word = SWAP_DATA(rdid.un.word);
            if(binfo->fc_rscn_id_cnt < FC_MAX_HOLD_RSCN) {
               for(j=0;j<binfo->fc_rscn_id_cnt;j++) {
                  if(binfo->fc_rscn_id_list[j] == rdid.un.word) {
                     goto skip_id;
                  }
               }
               binfo->fc_rscn_id_list[binfo->fc_rscn_id_cnt++] = rdid.un.word;
            }
            else {
               binfo->fc_flag |= FC_RSCN_DISCOVERY;
               goto out1;
            }
skip_id:
            cnt += (fc_handle_rscn(p_dev_ctl, &rdid));
            i -= sizeof(uint32);
         }

out1:
         fc_mem_put(binfo, MEM_BUF, (uchar * )mp);
   
         i = 1;
         /* free resources associated with this iocb and repost the ring buffers */
         if (!(binfo->fc_flag & FC_SLI2)) {
            for (i = 1; i < (int)iocb->ulpBdeCount; i++) {
               mp = fc_getvaddr(p_dev_ctl, rp, (uchar * )((ulong)iocb->un.cont[i].bdeAddress));
               if (mp) {
                  fc_mem_put(binfo, MEM_BUF, (uchar * )mp);
               }
            }
         }
         fc_mem_put(binfo, MEM_IOCB, (uchar * )xmitiq);
         if (binfo->fc_flag & FC_RSCN_DISCOVERY)
            goto out;
      }
      if (cnt == 0) {
         /* no need for nameserver login */
         fc_nextrscn(p_dev_ctl, fc_max_els_sent);
      }
      else
         fc_issue_ns_query(p_dev_ctl, (void *)0, (void *)0);

      return(0);
   }

   binfo->fc_flag &= ~FC_RSCN_MODE;
   binfo->fc_rscn_id_cnt = 0;

   /* This should turn off DELAYED ABTS for ELS timeouts */
   if ((mb = (MAILBOXQ * )fc_mem_get(binfo, MEM_MBOX))) {
      fc_set_slim(binfo, (MAILBOX * )mb, 0x052198, 0);
      if (issue_mb_cmd(binfo, (MAILBOX * )mb, MBX_NOWAIT) != MBX_BUSY) {
         fc_mem_put(binfo, MEM_MBOX, (uchar * )mb);
      }
   }
   /* Device Discovery completes */
   fc_log_printf_msg_vargs( binfo->fc_brd_no,
          &fc_msgBlk0249,                   /* ptr to msg structure */
           fc_mes0249,                      /* ptr to msg */
            fc_msgBlk0249.msgPreambleStr,   /* begin varargs */
              binfo->fc_flag);              /* end varargs */
   /* Fix up any changed RPIs in FCP IOCBs queued up a txq */
   fc_fcp_fix_txq(p_dev_ctl);


   if(FABRICTMO) {
      fc_clk_can(p_dev_ctl, FABRICTMO);
      FABRICTMO = 0;
   }

   if(clp[CFG_FCP_ON].a_current) {

      fc_restart_all_devices(p_dev_ctl);

      /* Call iodone for any commands that timed out previously */
      for (bp = p_dev_ctl->timeout_head; bp != NULL; ) {
         nextbp = bp->av_forw;
         bp->b_error = ETIMEDOUT;
         bp->b_flags |= B_ERROR;
         bp->b_resid = bp->b_bcount;
         FCSTATCTR.fcpScsiTmo++;
         fc_do_iodone(bp);
         bp = nextbp;
      }
      p_dev_ctl->timeout_count = 0;
      p_dev_ctl->timeout_head = NULL;
      /* Send down any saved FCP commands */
      fc_issue_cmd(p_dev_ctl);
   }
   return(0);
}       /* End fc_nextrscn */


_static_ int
fc_online(
fc_dev_ctl_t  * p_dev_ctl)
{
   FC_BRD_INFO * binfo;
   int  ipri;
   int  j;

   if(p_dev_ctl) {
      ipri = disable_lock(FC_LVL, &CMD_LOCK);
      binfo = &BINFO;
      if (!(binfo->fc_flag & FC_OFFLINE_MODE)) {
         unlock_enable(ipri, &CMD_LOCK);
         return(0);
      }
      /* Bring Adapter online */
      fc_log_printf_msg_vargs( binfo->fc_brd_no,
             &fc_msgBlk0458,                   /* ptr to msg structure */
              fc_mes0458,                      /* ptr to msg */
               fc_msgBlk0458.msgPreambleStr);  /* begin & end varargs */
      binfo->fc_flag &= ~FC_OFFLINE_MODE;

      fc_brdreset(p_dev_ctl);
      unlock_enable(ipri, &CMD_LOCK);
      fc_cfg_init(p_dev_ctl);
      return(0);
   }

   fc_diag_state = DDI_ONDI;

   /*
    * find the device in the dev_array if it is there
    */
   for (j = 0; j < MAX_FC_BRDS; j++) {
      p_dev_ctl = DD_CTL.p_dev[j];
      if (p_dev_ctl) {
         ipri = disable_lock(FC_LVL, &CMD_LOCK);
         binfo = &BINFO;
         if (!(binfo->fc_flag & FC_OFFLINE_MODE))
            continue;
         /* Bring Adapter online */
         fc_log_printf_msg_vargs( binfo->fc_brd_no,
                &fc_msgBlk0459,                   /* ptr to msg structure */
                 fc_mes0459,                      /* ptr to msg */
                  fc_msgBlk0459.msgPreambleStr);  /* begin & end varargs */
         binfo->fc_flag &= ~FC_OFFLINE_MODE;

         fc_brdreset(p_dev_ctl);
         unlock_enable(ipri, &CMD_LOCK);
         fc_cfg_init(p_dev_ctl);
         continue;
      }
   }
   return(0);
}       /* End fc_online */


_static_ int
fc_offline(
fc_dev_ctl_t  * p_dev_ctl)
{
   FC_BRD_INFO * binfo;
   int  ipri;
   int  j;

   if(p_dev_ctl) {
      ipri = disable_lock(FC_LVL, &CMD_LOCK);
      binfo = &BINFO;
      if (binfo->fc_flag & FC_OFFLINE_MODE) {
         unlock_enable(ipri, &CMD_LOCK);
         return(0);
      }
      /* Bring Adapter offline */
      fc_log_printf_msg_vargs( binfo->fc_brd_no,
             &fc_msgBlk0460,                   /* ptr to msg structure */
              fc_mes0460,                      /* ptr to msg */
               fc_msgBlk0460.msgPreambleStr);  /* begin & end varargs */

      fc_cfg_remove(p_dev_ctl);
      binfo->fc_flag |= FC_OFFLINE_MODE;

      unlock_enable(ipri, &CMD_LOCK);
      return(0);
   }
   fc_diag_state = DDI_OFFDI;

   /*
    * find the device in the dev_array if it is there
    */
   for (j = 0; j < MAX_FC_BRDS; j++) {
      p_dev_ctl = DD_CTL.p_dev[j];
      if (p_dev_ctl) {
         ipri = disable_lock(FC_LVL, &CMD_LOCK);
         binfo = &BINFO;
         if (binfo->fc_flag & FC_OFFLINE_MODE) {
            unlock_enable(ipri, &CMD_LOCK);
            continue;
         }
         /* Bring Adapter offline */
         fc_log_printf_msg_vargs( binfo->fc_brd_no,
                &fc_msgBlk0452,                   /* ptr to msg structure */
                 fc_mes0452,                      /* ptr to msg */
                  fc_msgBlk0452.msgPreambleStr);  /* begin & end varargs */

         fc_cfg_remove(p_dev_ctl);
         binfo->fc_flag |= FC_OFFLINE_MODE;
         unlock_enable(ipri, &CMD_LOCK);
         continue;
      }
   }
   return(0);
}       /* End fc_offline */


_static_ int
fc_attach(
int index,
uint32  *p_uio)         /* pointer to driver specific structure */
{
   fc_dev_ctl_t  * p_dev_ctl;
   FC_BRD_INFO * binfo;
   iCfgParam   * clp;
   int           rc, i;

   if ((p_dev_ctl = DD_CTL.p_dev[index]) == NULL) {
      rc = ENOMEM;
      return(rc);
   }

   binfo = &BINFO;
   fc_diag_state = DDI_ONDI;

   binfo->fc_brd_no = index;    /* FC board number */
   binfo->fc_p_dev_ctl = (uchar * )p_dev_ctl;
   binfo->nlptimer = 1;
   binfo->fc_fcpnodev.nlp_Rpi = 0xfffe;
   binfo->fc_nlpbind_start  = (NODELIST *)&binfo->fc_nlpbind_start;
   binfo->fc_nlpbind_end    = (NODELIST *)&binfo->fc_nlpbind_start;
   binfo->fc_nlpmap_start   = (NODELIST *)&binfo->fc_nlpmap_start;
   binfo->fc_nlpmap_end     = (NODELIST *)&binfo->fc_nlpmap_start;
   binfo->fc_nlpunmap_start = (NODELIST *)&binfo->fc_nlpunmap_start;
   binfo->fc_nlpunmap_end   = (NODELIST *)&binfo->fc_nlpunmap_start;

   /* Initialize current value of config parameters from default */
   clp = DD_CTL.p_config[binfo->fc_brd_no];
   for(i=0;i<NUM_CFG_PARAM;i++) {
      clp[i].a_current = clp[i].a_default;
   }
   /* Copy DDS from the config method and update configuration parameters */
   if (fc_get_dds(p_dev_ctl, p_uio)) {
      rc = EIO;
      return(rc);
   }
   binfo->fc_sli = (uchar)2; 
   clp[CFG_ZONE_RSCN].a_current = 1; /* ALWAYS force NS login on RSCN */

   /* config the device */
   if ((rc = fc_cfg_init(p_dev_ctl))) {
      return(rc);
   }
   return(0);
}


_static_ int
fc_detach(
int index)  /* device unit number */
{
   fc_dev_ctl_t      * p_dev_ctl;
   FC_BRD_INFO       * binfo;

   p_dev_ctl = DD_CTL.p_dev[index];
   binfo = &BINFO;
   if (p_dev_ctl == 0)
      return(0);


   if (!(binfo->fc_flag & FC_OFFLINE_MODE)) {
      /* Free the device resources */
      fc_cfg_remove(p_dev_ctl);
   }

   /* De-register the interrupt handler */
   if (p_dev_ctl->intr_inited) {
      i_clear(&IHS);
      p_dev_ctl->intr_inited = 0;
   }

   fc_unmemmap(p_dev_ctl);
   return(0);
}


/*****************************************************************************/
/*
 * NAME:     fc_cfg_init
 *
 * FUNCTION: perform CFG_INIT function. Initialize the device control 
 *           structure and get the adapter VPD data.
 *
 * EXECUTION ENVIRONMENT: process only
 *
 * CALLED FROM:
 *      fc_config
 *
 * INPUT:
 *      p_dev_ctl       - pointer to the dev_ctl area
 *
 * RETURNS:  
 *      0 - OK
 *      EEXIST - device name in use (from ns_attach)
 *      EINVAL - invalid parameter was passed
 *      EIO - permanent I/O error
 */
/*****************************************************************************/
_static_ int
fc_cfg_init(
fc_dev_ctl_t  *p_dev_ctl)  /* pointer to the device control area */
{
   int  rc;             /* return code */
   int  i;
   FC_BRD_INFO * binfo;
   iCfgParam   * clp;

   binfo = &BINFO;
   clp = DD_CTL.p_config[binfo->fc_brd_no];
   p_dev_ctl->ctl_correlator = (void * ) & DD_CTL;

   for (i = 0; i < NLP_MAXPAN; i++) {
      p_dev_ctl->adapter_state[i] = CLOSED;
   }

   if ((rc = fc_pcimap(p_dev_ctl))) {
      return(rc);
   }

   if ((rc = fc_memmap(p_dev_ctl))) {
      return(rc);
   }

   /* offset from beginning of SLIM */
   BINFO.fc_mboxaddr    = 0;

   BINFO.fc_mbox_active = 0;
   BINFO.fc_ns_retry = 0;
   BINFO.fc_process_LA = 0;
   BINFO.fc_edtov = FF_DEF_EDTOV;
   BINFO.fc_ratov = FF_DEF_RATOV;
   BINFO.fc_altov = FF_DEF_ALTOV;
   BINFO.fc_arbtov = FF_DEF_ARBTOV;

   /* offset from beginning of register space */
   BINFO.fc_HAregaddr   = (sizeof(uint32) * HA_REG_OFFSET);
   BINFO.fc_FFregaddr   = (sizeof(uint32) * CA_REG_OFFSET);
   BINFO.fc_STATregaddr = (sizeof(uint32) * HS_REG_OFFSET);
   BINFO.fc_HCregaddr   = (sizeof(uint32) * HC_REG_OFFSET);
   BINFO.fc_BCregaddr   = (sizeof(uint32) * BC_REG_OFFSET);


   /* save the dev_ctl address in the NDD correlator field */
   NDD.ndd_name = DDS.logical_name;/* point to the name contained in the dds */
   NDD.ndd_alias = DDS.dev_alias;  /* point to the name contained in the dds */



   binfo->fc_ring[FC_IP_RING].fc_tx.q_max = 
      (ushort)clp[CFG_XMT_Q_SIZE].a_current;

   p_dev_ctl->iostrat_event = EVENT_NULL;
   p_dev_ctl->iostrat_head = NULL;
   p_dev_ctl->iostrat_tail = NULL;

   /* 
     * Perform any device-specific initialization necessary at the
     * CFG_INIT time. If there is any error during the device initialization,
     * the CFG_INIT will fail. Also get VPD data.
     */
   if ((rc = fc_ffinit(p_dev_ctl))) {
      return(rc);
   }

   /* Now setup physical address */
   fc_bcopy(binfo->fc_portname.IEEE, p_dev_ctl->phys_addr, 6);

   return(0);
}       /* End fc_cfg_init */


/*****************************************************************************/
/*
 * NAME:     fc_cfg_remove
 *
 * FUNCTION: Remove the device resources that have been allocated during
 *           CFG_INIT configuration time.
 *
 * EXECUTION ENVIRONMENT: process only
 *
 * NOTES:
 *
 * CALLED FROM:
 *      fc_config
 *
 * INPUT:
 *      p_dev_ctl - address of a pointer to the dev control structure
 *
 * RETURNS:  
 *      none.
 */
/*****************************************************************************/
_static_ void
fc_cfg_remove(
fc_dev_ctl_t    *p_dev_ctl)    /* point to the dev_ctl area */
{
   fc_free_rpilist(p_dev_ctl, 0);

   /* Release the watchdog timers and disable board interrupts */
   fc_ffcleanup(p_dev_ctl);

   fc_free_buffer(p_dev_ctl);           /* free device buffers */

   fc_brdreset(p_dev_ctl);

}       /* End fc_cfg_remove */


/*****************************************************************************/
/*
 * NAME:     fc_ffcleanup
 *
 * EXECUTION ENVIRONMENT: process only
 *
 * CALLED FROM:
 *      CFG_TERM
 *
 * INPUT:
 *      p_dev_ctl       - pointer to the dev_ctl area.
 *
 * RETURNS:  
 *      none
 */
/*****************************************************************************/
_static_ void
fc_ffcleanup(
fc_dev_ctl_t    *p_dev_ctl)     /* pointer to the dev_ctl area */
{
   int  i;
   RING  * rp;
   FC_BRD_INFO * binfo;
   void *ioa;
   MAILBOX       * mb;

   binfo = &BINFO;
   binfo->fc_process_LA = 0;

   /* Disable all but the mailbox interrupt */
   ioa = (void *)FC_MAP_IO(&binfo->fc_iomap_io);  /* map in io registers */
   WRITE_CSR_REG(binfo, FC_HC_REG(binfo, ioa), HC_MBINT_ENA);
   FC_UNMAP_MEMIO(ioa);

   /* Issue unreg_login command to logout all nodes */
   if (p_dev_ctl->init_eventTag) {
      /* Get a buffer for mailbox command */
      if ((mb = (MAILBOX * )fc_mem_get(binfo, MEM_MBOX)) == NULL) {
      } else {
         fc_unreg_login(binfo, 0xffff, (MAILBOX * )mb);
         if (issue_mb_cmd(binfo, (MAILBOX * )mb, MBX_NOWAIT) != MBX_BUSY)
            fc_mem_put(binfo, MEM_MBOX, (uchar * )mb);
      }
   }

   ioa = (void *)FC_MAP_IO(&binfo->fc_iomap_io);  /* map in io registers */
   /* Clear all interrupt enable conditions */
   WRITE_CSR_REG(binfo, FC_HC_REG(binfo, ioa), 0);
   FC_UNMAP_MEMIO(ioa);

   for (i = 0; i < binfo->fc_ffnumrings; i++) {
      rp = &binfo->fc_ring[i];
      /* Clear the transmit watchdog timer */
      if (rp->fc_wdt_inited) {
         if(RINGTMO) {
            fc_clk_can(p_dev_ctl, RINGTMO);
            RINGTMO = 0;
         }
         rp->fc_wdt_inited = 0;
      }
   }

   if(MBOXTMO) {
      fc_clk_can(p_dev_ctl, MBOXTMO);
      MBOXTMO = 0;
   }
   if(FABRICTMO) {
      fc_clk_can(p_dev_ctl, FABRICTMO);
      FABRICTMO = 0;
   }

   fc_flush_rscn_defer(p_dev_ctl);

   fc_flush_clk_set(p_dev_ctl, fc_delay_timeout);

   fc_flush_clk_set(p_dev_ctl, lpfc_scsi_selto_timeout);

}       /* End fc_ffcleanup */


/*****************************************************************************/
/*
 * NAME:     fc_start
 *
 * FUNCTION: Initialize and activate the adapter.
 *
 * EXECUTION ENVIRONMENT: process or interrupt
 *
 * CALLED FROM:
 *      fc_config
 *
 * INPUT:
 *      p_dev_ctl       - pointer to the dev_ctl area.
 *
 * RETURNS:
 *      NONE
 */
/*****************************************************************************/

_static_ void
fc_start(
fc_dev_ctl_t  *p_dev_ctl)       /* pointer to the dev_ctl area */
{
   uint32        i, j;
   FC_BRD_INFO * binfo;
   iCfgParam   * clp;
   void        * ioa;
   RING        * rp;

   /* Activate the adapter and allocate all the resources needed */

   binfo = &BINFO;
   clp = DD_CTL.p_config[binfo->fc_brd_no];

   /* Enable appropriate host interrupts */
   i = (uint32) (HC_MBINT_ENA | HC_ERINT_ENA);
   if (binfo->fc_ffnumrings > 0)
      i |= HC_R0INT_ENA;
   if (binfo->fc_ffnumrings > 1)
      i |= HC_R1INT_ENA;
   if (binfo->fc_ffnumrings > 2)
      i |= HC_R2INT_ENA;
   if (binfo->fc_ffnumrings > 3)
      i |= HC_R3INT_ENA;

   ioa = (void *)FC_MAP_IO(&binfo->fc_iomap_io);  /* map in io registers */
   WRITE_CSR_REG(binfo, FC_HC_REG(binfo, ioa), i);
   FC_UNMAP_MEMIO(ioa);

   for (i = 0; i < (uint32)binfo->fc_ffnumrings; i++) {
      /* Initialize / post buffers to ring */
      fc_setup_ring(p_dev_ctl, i);

      if (i == FC_ELS_RING) {
         /* Now post receive buffers to the ring */
         rp = &binfo->fc_ring[i];
         for (j = 0; j < 64; j++)
            fc_post_buffer(p_dev_ctl, rp, 2);
      }
   }

   clp = DD_CTL.p_config[binfo->fc_brd_no];
   if(clp[CFG_NETWORK_ON].a_current) {
      rp = &binfo->fc_ring[FC_IP_RING];
      i = clp[CFG_POST_IP_BUF].a_current;
      while(i) {
         fc_post_mbuf(p_dev_ctl, rp, 2);
         i -= 2;
      }
   }

   /* set up the watchdog timer control structure section */
   binfo->fc_fabrictmo = FF_DEF_RATOV + 1;

}       /* End fc_start */


_static_ void
fc_process_reglogin(
fc_dev_ctl_t  *p_dev_ctl,       /* pointer to the dev_ctl area */
NODELIST      *ndlp)
{
   node_t  * node_ptr;
   RING    * rp;
   FC_BRD_INFO   * binfo;
   iCfgParam     * clp;

   binfo = &BINFO;
   clp = DD_CTL.p_config[binfo->fc_brd_no];

   ndlp->nlp_flag &= ~NLP_REG_INP;
   if (ndlp->nlp_DID == Fabric_DID) {
      ndlp->nlp_flag &= ~NLP_FARP_SND;
      ndlp->nlp_action &= ~NLP_DO_ADDR_AUTH;
   } else {
      /* If we are an FCP node, update the rpi */
      if (ndlp->nlp_type & NLP_FCP_TARGET) {
         if ((node_ptr = (node_t * )ndlp->nlp_targetp) != NULL) {
            node_ptr->rpi = (ushort)ndlp->nlp_Rpi;
            node_ptr->last_good_rpi = (ushort)ndlp->nlp_Rpi;
            node_ptr->nlp = ndlp;
            node_ptr->flags &= ~FC_NODEV_TMO;
            ndlp->nlp_flag &= ~NLP_NODEV_TMO;
         }
         else {
            int  dev_index;

            dev_index = INDEX(ndlp->id.nlp_pan, ndlp->id.nlp_sid);
            node_ptr =  binfo->device_queue_hash[dev_index].node_ptr;
            if(node_ptr) {
               /* This is a new device that entered the loop */
               node_ptr->nlp = ndlp;
               node_ptr->rpi = (ushort)ndlp->nlp_Rpi;
               node_ptr->last_good_rpi = (ushort)ndlp->nlp_Rpi;
               node_ptr->scsi_id = dev_index;
               ndlp->nlp_targetp = (uchar *)node_ptr;
               node_ptr->flags &= ~FC_NODEV_TMO;
               ndlp->nlp_flag &= ~NLP_NODEV_TMO;
            }
         }
      }

      if((ndlp->nlp_DID & CT_DID_MASK) == CT_DID_MASK)
         ndlp->nlp_state = NLP_LOGIN;

      /* HBA Mgmt */
      if(ndlp->nlp_DID == FDMI_DID) {
         ndlp->nlp_state = NLP_LOGIN;
         return;
      }

      /* If we are a NameServer, go to next phase */
      if (ndlp->nlp_DID == NameServer_DID) {
         int        fabcmd;

         ndlp->nlp_state = NLP_LOGIN;

         if(binfo->fc_ffstate == FC_READY) {
            fabcmd = SLI_CTNS_GID_FT;
         }
         else {
            fabcmd = SLI_CTNS_RFT_ID;
         }

         /* Issue RFT_ID / GID_FT to Nameserver */
         if (fc_ns_cmd(p_dev_ctl, ndlp, fabcmd)) {
            /* error so start discovery */
            /* Done with NameServer for now, but keep logged in */
            ndlp->nlp_action &= ~NLP_DO_RSCN;

            /* Fire out PLOGIs on nodes marked for discovery */
            if ((binfo->fc_nlp_cnt <= 0) && 
                !(binfo->fc_flag & FC_NLP_MORE)) {
                binfo->fc_nlp_cnt = 0;
                if ((binfo->fc_ffstate == FC_READY) &&
                   (binfo->fc_flag & FC_RSCN_MODE)) {
                   fc_nextrscn(p_dev_ctl, fc_max_els_sent);
                }
                else {
                   fc_nextnode(p_dev_ctl, ndlp);
                }
            }
            else {
                fc_nextnode(p_dev_ctl, ndlp);
            }
            ndlp->nlp_action &= ~NLP_DO_ADDR_AUTH;
         }
         return;
      }

      /* If we are in the middle of Discovery */
      if ((ndlp->nlp_type & NLP_FCP_TARGET) ||
         (ndlp->nlp_action & NLP_DO_DISC_START) ||
         (ndlp->nlp_action & NLP_DO_ADDR_AUTH) ||
         (ndlp->nlp_action & NLP_DO_RSCN) ||
         (ndlp->nlp_action & NLP_DO_SCSICMD) ||
         (binfo->fc_flag & FC_PT2PT) ||
         (ndlp->nlp_portname.nameType != NAME_IEEE)) {

         ndlp->nlp_flag &= ~NLP_FARP_SND;
         ndlp->nlp_action &= ~NLP_DO_ADDR_AUTH;
         if((!(binfo->fc_flag & FC_PT2PT)) && (ndlp->nlp_action == 0)) {
            if(binfo->fc_ffstate == FC_READY) {
               ndlp->nlp_action |= NLP_DO_RSCN;
            }
            else {
               ndlp->nlp_action |= NLP_DO_DISC_START;
            }
         }
         if(clp[CFG_FCP_ON].a_current) {
            ndlp->nlp_state = NLP_PRLI;
            if((ndlp->nlp_flag & NLP_RCV_PLOGI) &&
               (!(ndlp->nlp_action) || (ndlp->nlp_flag & NLP_REQ_SND)) &&
               !(binfo->fc_flag & FC_PT2PT)) {
               ndlp->nlp_state = NLP_LOGIN;
            }
            else {
               if((ndlp->nlp_DID & Fabric_DID_MASK) != Fabric_DID_MASK) {
                  fc_els_cmd(binfo, ELS_CMD_PRLI,
                    (void *)((ulong)ndlp->nlp_DID), (uint32)0, (ushort)0, ndlp);
               }
               else
                  fc_nextnode(p_dev_ctl, ndlp);
            }
         } else {
            /* establish a new exchange for login registration */
            if ((ndlp->nlp_Xri == 0) && 
                (ndlp->nlp_type & NLP_IP_NODE) && 
                ((ndlp->nlp_DID & CT_DID_MASK) != CT_DID_MASK) &&
                !(ndlp->nlp_flag & NLP_RPI_XRI)) {
                   ndlp->nlp_flag |= NLP_RPI_XRI;
                   rp = &binfo->fc_ring[FC_ELS_RING];
                   fc_create_xri(binfo, rp, ndlp);
            }
            if(!(ndlp->nlp_flag & NLP_RCV_PLOGI))
               fc_nextnode(p_dev_ctl, ndlp);
         }
      } else {
         ndlp->nlp_flag &= ~NLP_FARP_SND;
         ndlp->nlp_action &= ~NLP_DO_ADDR_AUTH;
         /* establish a new exchange for Nport login registration */
         if ((ndlp->nlp_Xri == 0) && 
             ((ndlp->nlp_DID & CT_DID_MASK) != CT_DID_MASK) &&
             !(ndlp->nlp_flag & NLP_RPI_XRI)) {
            ndlp->nlp_flag |= NLP_RPI_XRI;
            rp = &binfo->fc_ring[FC_ELS_RING];
            fc_create_xri(binfo, rp, ndlp);    /* IP ONLY */
         }
      }
      ndlp->nlp_flag &= ~NLP_RCV_PLOGI;
   }
   return;
}

_static_ int
fc_snd_scsi_req(
fc_dev_ctl_t *p_dev_ctl,
NAME_TYPE *wwn, 
MATCHMAP  *bmp, 
DMATCHMAP  *fcpmp,
DMATCHMAP *omatp,
uint32     count,
struct dev_info *dev_ptr)
{
   FC_BRD_INFO *binfo; 
   NODELIST    * ndlp;
   RING        * rp;
   IOCBQ       * temp;
   IOCB        * cmd;
   ULP_BDE64   * bpl;
   FCP_CMND    * inqcmnd;
   fc_buf_t    * fcptr;
   node_t      * map_node_ptr;
   struct dev_info * map_dev_ptr;
   uint32        did;
   fc_lun_t      lun;
   int           i;

   binfo = &BINFO;
   if(((ndlp = fc_findnode_wwpn(binfo, NLP_SEARCH_ALL, wwn)) == 0) ||
      (!(binfo->fc_flag & FC_SLI2))) {  /* MUST be SLI2 */
      return(EACCES);
   }

   if(ndlp->nlp_flag & NLP_REQ_SND) {
      return(ENODEV);
   }

   if(ndlp->nlp_state <= NLP_LOGIN) {
      if ((ndlp->nlp_DID == binfo->fc_myDID) || 
          (ndlp->nlp_DID & Fabric_DID_MASK)) {
         return(ENODEV);
      }
      ndlp->nlp_action |= NLP_DO_SCSICMD;
      if((ndlp->nlp_state == NLP_LOGIN) && ndlp->nlp_Rpi) {
         /* Need to send PRLI */
         fc_els_cmd(binfo, ELS_CMD_PRLI,
            (void *)((ulong)ndlp->nlp_DID), (uint32)0, (ushort)0, ndlp);
      }
      else {
         /* Need to send PLOGI */
         did = ndlp->nlp_DID;
         if(did == 0) {
            did = ndlp->nlp_oldDID;
         }
         if(!(ndlp->nlp_flag & NLP_NS_REMOVED)) {
            fc_els_cmd(binfo, ELS_CMD_PLOGI,
               (void *)((ulong)did), (uint32)0, (ushort)0, ndlp);
         }
      }
      return(ENODEV);
   }

   inqcmnd = (FCP_CMND *)fcpmp->dfc.virt;
   lun = ((inqcmnd->fcpLunMsl >> FC_LUN_SHIFT) & 0xff);

   map_node_ptr = 0;
   map_dev_ptr = 0;

   if (ndlp->nlp_type & NLP_SEED_MASK) {
      /* If this is a mapped target, check qdepth limits */
      i = INDEX(ndlp->id.nlp_pan, ndlp->id.nlp_sid);
      if ((map_node_ptr = binfo->device_queue_hash[i].node_ptr) != NULL) {

            if (map_node_ptr->tgt_queue_depth && 
               (map_node_ptr->tgt_queue_depth == map_node_ptr->num_active_io))
               return(ENODEV);

            if ((map_dev_ptr = fc_find_lun(binfo, i, lun))) {
               if ((map_dev_ptr->active_io_count >= map_dev_ptr->fcp_cur_queue_depth) ||
                  (map_dev_ptr->stop_send_io))
                  return(ENODEV);
         }
      }
   }

   rp = &binfo->fc_ring[FC_FCP_RING];
   if ((temp = (IOCBQ * )fc_mem_get(binfo, MEM_IOCB)) == NULL) {
      return(EACCES);
   }

   fc_bzero((void *)dev_ptr, sizeof(dev_ptr));
   dev_ptr->lun_id = lun;
   dev_ptr->opened = TRUE;
   dev_ptr->fcp_lun_queue_depth = 1;
   dev_ptr->fcp_cur_queue_depth = 1;
   dev_ptr->queue_state = ACTIVE_PASSTHRU;
   dev_ptr->pend_head = (T_SCSIBUF *)map_node_ptr;
   dev_ptr->pend_tail = (T_SCSIBUF *)map_dev_ptr;

   fcptr = (fc_buf_t *)fcpmp->dfc.virt;
   fcptr->dev_ptr = dev_ptr;
   fcptr->phys_adr = (char *)fcpmp->dfc.phys;
   fcptr->sc_bufp = (T_SCSIBUF *)omatp;
   fcptr->flags = 0;
   /* set up an iotag so we can match the completion iocb */
   for (i = 0; i < MAX_FCP_CMDS; i++) {
      fcptr->iotag = rp->fc_iotag++;
      if (rp->fc_iotag >= MAX_FCP_CMDS)
         rp->fc_iotag = 1;
      if (binfo->fc_table->fcp_array[fcptr->iotag] == 0)
         break;
   }
   if (i >= MAX_FCP_CMDS) {
      /* No more command slots available, retry later */
      fc_mem_put(binfo, MEM_IOCB, (uchar * )temp);
      return(EACCES);
   }

   fc_bzero((void *)temp, sizeof(IOCBQ));
   cmd = &temp->iocb;

   bpl = (ULP_BDE64 * )bmp->virt;

   cmd->un.fcpi64.bdl.ulpIoTag32 = (uint32)0;
   cmd->un.fcpi64.bdl.addrHigh = (uint32)putPaddrHigh(bmp->phys);
   cmd->un.fcpi64.bdl.addrLow = (uint32)putPaddrLow(bmp->phys);
   cmd->un.fcpi64.bdl.bdeFlags = BUFF_TYPE_BDL;
   cmd->ulpBdeCount = 1;
   fcptr->bmp = bmp;
   temp->bpl = (uchar *)0;

   cmd->ulpContext = ndlp->nlp_Rpi;
   cmd->ulpIoTag = fcptr->iotag;
   /*
    * if device is FCP-2 device, set the following bit that says
    * to run the FC-TAPE protocol.
    */
   if (ndlp->id.nlp_fcp_info & NLP_FCP_2_DEVICE) {
      cmd->ulpFCP2Rcvy = 1;
   }
   cmd->ulpClass = (ndlp->id.nlp_fcp_info & 0x0f);
   cmd->ulpOwner = OWN_CHIP;

   /* Hardcode 30 second timeout for I/O to complete */
   curtime(&fcptr->timeout);
   cmd->ulpRsvdByte = fc_inq_sn_tmo;
   fcptr->timeout = ((ulong)fcptr->timeout + (31 * fc_ticks_per_second));

   switch(fcptr->fcp_cmd.fcpCntl3) {
   case READ_DATA:
      /* Set up for SCSI read */
      cmd->ulpCommand = CMD_FCP_IREAD64_CR;
      cmd->ulpPU = PARM_READ_CHECK;
      cmd->un.fcpi.fcpi_parm = count;
      cmd->un.fcpi64.bdl.bdeSize = ((omatp->dfc_flag+2) * sizeof(ULP_BDE64));
      cmd->ulpBdeCount = 1;
      break;

   case WRITE_DATA:
      /* Set up for SCSI write */
      cmd->ulpCommand = CMD_FCP_IWRITE64_CR;
      cmd->un.fcpi64.bdl.bdeSize = ((omatp->dfc_flag+2) * sizeof(ULP_BDE64));
      cmd->ulpBdeCount = 1;
      break;
   default:
      /* Set up for SCSI command */
      cmd->ulpCommand = CMD_FCP_ICMND64_CR;
      cmd->un.fcpi64.bdl.bdeSize = (2 * sizeof(ULP_BDE64));
      cmd->ulpBdeCount = 1;
      break;
   }

   cmd->ulpLe = 1;
   /* Queue cmd chain to last iocb entry in xmit queue */
   if (rp->fc_tx.q_first == NULL) {
      rp->fc_tx.q_first = (uchar * )temp;
   } else {
      ((IOCBQ * )(rp->fc_tx.q_last))->q  = (uchar * )temp;
   }
   rp->fc_tx.q_last = (uchar * )temp;
   rp->fc_tx.q_cnt++;

   fc_enq_fcbuf_active(rp, fcptr);

   if(map_dev_ptr)
      map_dev_ptr->active_io_count++;
   if(map_node_ptr)
      map_node_ptr->num_active_io++;
   dev_ptr->active_io_count++;
   FCSTATCTR.fcpCmd++;

   issue_iocb_cmd(binfo, rp, 0);
   return(0);
}


/******************************************************************************
* Function name : fc_parse_binding_entry
*
* Description   : Parse binding entry for WWNN & WWPN
*
* ASCII Input string example: 2000123456789abc:lpfc1t0
* 
* Return        :  0              = Success
*                  Greater than 0 = Binding entry syntax error. SEE defs
*                                   FC_SYNTAX_ERR_XXXXXX.
******************************************************************************/
_static_ int
fc_parse_binding_entry( fc_dev_ctl_t *p_dev_ctl,
                       uchar *inbuf, uchar *outbuf,
                       int in_size, int out_size,
                       int bind_type,
                       unsigned int *sum, int entry, int *lpfc_num)
{
  int         brd; 
  int         c1, cvert_cnt, sumtmp;

  FC_BRD_INFO * binfo = &BINFO;

  char ds_lpfc[] = "lpfc";

  *lpfc_num = -1;  

  /* Parse 16 digit ASC hex address */
  if( bind_type == FC_BIND_DID)  outbuf++;
  cvert_cnt = fc_asc_seq_to_hex( p_dev_ctl, in_size, out_size, (char *)inbuf, (char *)outbuf);
  if(cvert_cnt < 0)
    return(FC_SYNTAX_ERR_ASC_CONVERT);
  inbuf += (ulong)cvert_cnt;
  
  /* Parse colon */
  if(*inbuf++ != ':')
    return(FC_SYNTAX_ERR_EXP_COLON);

  /* Parse lpfc */
  if(fc_strncmp( (char *)inbuf, ds_lpfc, (sizeof(ds_lpfc)-1)))
    return(FC_SYNTAX_ERR_EXP_LPFC);
  inbuf += sizeof(ds_lpfc)-1;

  /* Parse lpfc number */
  /* Get 1st lpfc digit */
  c1 = *inbuf++;
  if(fc_is_digit(c1) == 0) 
    goto  err_lpfc_num;
  sumtmp = c1 - 0x30;

  /* Get 2nd lpfc digit */
  c1 = *inbuf;
  if(fc_is_digit(c1) == 0) 
    goto  convert_instance;
  inbuf++;
  sumtmp = (sumtmp * 10) + c1 - 0x30;
  if((sumtmp < 0) || (sumtmp > 15))
    goto err_lpfc_num;
  goto  convert_instance;

err_lpfc_num:

  return(FC_SYNTAX_ERR_INV_LPFC_NUM);

  /* Convert from ddi instance number to adapter number */
convert_instance:

  for(brd = 0; brd < MAX_FC_BRDS; brd++) {
    if(fcinstance[brd] == sumtmp)
      break;
  }
  if(binfo->fc_brd_no != brd) {
    /* Skip this entry */
    return(FC_SYNTAX_OK_BUT_NOT_THIS_BRD);
  }


  /* Parse 't' */
  if(*inbuf++ != 't')
    return(FC_SYNTAX_ERR_EXP_T);

  /* Parse target number */
  /* Get 1st target digit */
  c1 = *inbuf++;
  if(fc_is_digit(c1) == 0) 
    goto  err_target_num;
  sumtmp = c1 - 0x30;

  /* Get 2nd target digit */
  c1 = *inbuf;
  if(fc_is_digit(c1) == 0) 
    goto  check_for_term;
  inbuf++;
  sumtmp = (sumtmp * 10) + c1 - 0x30;

  /* Get 3nd target digit */
  c1 = *inbuf;
  if(fc_is_digit(c1) == 0) 
    goto  check_for_term;
  inbuf++;
  sumtmp = (sumtmp * 10) + c1 - 0x30;
  if((sumtmp < 0) || (sumtmp > 999))
    goto err_target_num;
  goto  check_for_term;

err_target_num:
  return(FC_SYNTAX_ERR_INV_TARGET_NUM);

  /* Test that input string in NULL terminated - End of input */
check_for_term:

  if(*inbuf != 0)
    return(FC_SYNTAX_ERR_EXP_NULL_TERM);


  *sum = sumtmp;
  return(FC_SYNTAX_OK); /* Success */
} /* fc_parse_binding_entry */

void
issue_report_lun(
fc_dev_ctl_t *pd,
void *l1,
void *l2)
{
  FC_BRD_INFO   * binfo = &pd->info;
  dvi_t     * di = (dvi_t *)l1;
  RING      * rp;
  fc_buf_t  * fcptr;
  IOCBQ     * temp;
  IOCB      * cmd;
  ULP_BDE64 * bpl;
  MATCHMAP  * bmp;
  MBUF_INFO * mbufp;
  node_t    * nodep;
  int         i, tmo;

  rp = &binfo->fc_ring[FC_FCP_RING];
  nodep = di->nodep;

  mbufp = (MBUF_INFO * )fc_mem_get(binfo, MEM_IOCB);
  if (mbufp == NULL) {
    nodep->rptlunstate = REPORT_LUN_COMPLETE;
    return;
  } 
  mbufp->virt = 0;
  mbufp->phys = 0;
  mbufp->flags  = FC_MBUF_DMA;
  mbufp->align = (int)4096;
  mbufp->size = 4096;

  if (nodep->virtRptLunData == 0) {
     fc_malloc(pd, mbufp);
     if (mbufp->phys == NULL) {
       fc_mem_put(binfo, MEM_IOCB, (uchar * )mbufp);
       nodep->rptlunstate = REPORT_LUN_COMPLETE;
       return;
     }
  } else {
     mbufp->phys = nodep->physRptLunData;
     mbufp->virt = nodep->virtRptLunData;
  }

  if ((fcptr = fc_deq_fcbuf(di)) == NULL) {
    if (nodep->virtRptLunData == 0)
       fc_free(pd, mbufp);
    fc_mem_put(binfo, MEM_IOCB, (uchar * )mbufp);
    nodep->rptlunstate = REPORT_LUN_COMPLETE;
    return;
  }

  if ((temp = (IOCBQ * )fc_mem_get(binfo, MEM_IOCB)) == NULL) {
    if (nodep->virtRptLunData == 0)
       fc_free(pd, mbufp);
    fc_mem_put(binfo, MEM_IOCB, (uchar * )mbufp);
    fc_enq_fcbuf(fcptr);
    nodep->rptlunstate = REPORT_LUN_COMPLETE;
    return;
  }

  fc_bzero((void *)fcptr, sizeof(FCP_CMND) + sizeof(FCP_RSP));

  /*
   * Save the MBUF pointer.
   * Buffer will be freed by handle_fcp_event().
   */
  fcptr->sc_bufp = (void *)mbufp;

  /*
   * Setup SCSI command block in FCP payload
   */
  fcptr->fcp_cmd.fcpCdb[0]= 0xA0;   /* SCSI Report Lun Command */
  
  fcptr->fcp_cmd.fcpCdb[8]= 0x10;
  fcptr->fcp_cmd.fcpCntl3 = READ_DATA;
  fcptr->fcp_cmd.fcpDl = SWAP_DATA(RPTLUN_MIN_LEN);

  /* 
   * set up an iotag so we can match the completion iocb 
   */
  for (i = 0; i < MAX_FCP_CMDS; i++) {
    fcptr->iotag = rp->fc_iotag++;
    if (rp->fc_iotag >= MAX_FCP_CMDS)
       rp->fc_iotag = 1;
    if (binfo->fc_table->fcp_array[fcptr->iotag] == 0)
       break;
  }
  if (i >= MAX_FCP_CMDS) {
    /* 
     * No more command slots available 
     */
    if (nodep->virtRptLunData == 0)
       fc_free(pd, mbufp);
    fc_mem_put(binfo, MEM_IOCB, (uchar * )mbufp);
    fc_mem_put(binfo, MEM_IOCB, (uchar * )temp);
    fc_enq_fcbuf(fcptr);
    nodep->rptlunstate = REPORT_LUN_COMPLETE;
    return;
  }

  fc_bzero((void *)temp, sizeof(IOCBQ));  
  cmd = &temp->iocb;
  temp->q = NULL;

  /* 
   * Allocate buffer for Buffer ptr list 
   */
  if ((bmp = (MATCHMAP * )fc_mem_get(binfo, MEM_BPL)) == 0) {
    if (nodep->virtRptLunData == 0)
       fc_free(pd, mbufp);
    fc_mem_put(binfo, MEM_IOCB, (uchar * )mbufp);
    fc_mem_put(binfo, MEM_IOCB, (uchar * )temp);
    fc_enq_fcbuf(fcptr);
    nodep->rptlunstate = REPORT_LUN_COMPLETE;
    return;
  }

  bpl = (ULP_BDE64 * )bmp->virt;
  bpl->addrHigh = PCIMEM_LONG((uint32)putPaddrHigh(GET_PAYLOAD_PHYS_ADDR(fcptr)));
  bpl->addrLow = PCIMEM_LONG((uint32)putPaddrLow(GET_PAYLOAD_PHYS_ADDR(fcptr)));
  bpl->tus.f.bdeSize = sizeof(FCP_CMND);
  bpl->tus.f.bdeFlags = BUFF_USE_CMND;
  bpl->tus.w = PCIMEM_LONG(bpl->tus.w);
  bpl++;
  bpl->addrHigh = PCIMEM_LONG((uint32)putPaddrHigh(GET_PAYLOAD_PHYS_ADDR(fcptr)+sizeof(FCP_CMND)));
  bpl->addrLow = PCIMEM_LONG((uint32)putPaddrLow(GET_PAYLOAD_PHYS_ADDR(fcptr)+sizeof(FCP_CMND)));
  bpl->tus.f.bdeSize = sizeof(FCP_RSP);
  bpl->tus.f.bdeFlags = (BUFF_USE_CMND | BUFF_USE_RCV);
  bpl->tus.w = PCIMEM_LONG(bpl->tus.w);
  bpl++;

  cmd->un.fcpi64.bdl.ulpIoTag32 = (uint32)0;
  cmd->un.fcpi64.bdl.addrHigh = (uint32)putPaddrHigh(bmp->phys);
  cmd->un.fcpi64.bdl.addrLow = (uint32)putPaddrLow(bmp->phys);
  cmd->un.fcpi64.bdl.bdeSize = (2 * sizeof(ULP_BDE64));
  cmd->un.fcpi64.bdl.bdeFlags = BUFF_TYPE_BDL;
  cmd->ulpBdeCount = 1;
  fcptr->bmp = bmp;
  temp->bpl = (uchar *)0;

  cmd->ulpContext = nodep->rpi;
  cmd->ulpIoTag = fcptr->iotag;

  /*
   * if device is FCP-2 device, set the following bit that says
   * to run the FC-TAPE protocol.
   */
  if (nodep->nlp->id.nlp_fcp_info & NLP_FCP_2_DEVICE) {
    cmd->ulpFCP2Rcvy = 1;
  }
  cmd->ulpClass = (nodep->nlp->id.nlp_fcp_info & 0x0f);
  cmd->ulpOwner = OWN_CHIP;

  /* 
   * Hardcode 2*RATOV second timeout for I/O to complete 
   */
  tmo = (2 * binfo->fc_ratov);
  curtime(&fcptr->timeout);
  cmd->ulpRsvdByte = tmo;
  tmo++; /* Make scsi timeout longer then cmd tmo */
   fcptr->timeout = ((ulong)fcptr->timeout + (tmo * fc_ticks_per_second));

  /*
   * Read Data
   */
  cmd->ulpCommand = CMD_FCP_IREAD64_CR;
  cmd->ulpPU = PARM_READ_CHECK;
  cmd->un.fcpi.fcpi_parm = RPTLUN_MIN_LEN;

  bpl->addrHigh = PCIMEM_LONG((uint32)putPaddrHigh(mbufp->phys));
  bpl->addrLow = PCIMEM_LONG((uint32)putPaddrLow(mbufp->phys));
  bpl->tus.f.bdeSize = RPTLUN_MIN_LEN;
  bpl->tus.f.bdeFlags = BUFF_USE_RCV;
  bpl->tus.w = PCIMEM_LONG(bpl->tus.w);
  bpl++;

  cmd->un.fcpi64.bdl.bdeSize += sizeof(ULP_BDE64);
  cmd->ulpBdeCount = 1;

  cmd->ulpLe = 1;

  /* 
   * Queue cmd chain to last iocb entry in xmit queue 
   */
  if (rp->fc_tx.q_first == NULL) {
    rp->fc_tx.q_first = (uchar * )temp;
  } else {
    ((IOCBQ * )(rp->fc_tx.q_last))->q  = (uchar * )temp;
  }
  rp->fc_tx.q_last = (uchar * )temp;
  rp->fc_tx.q_cnt++;

  fc_enq_fcbuf_active(rp, fcptr);
  fcptr->flags |= FCBUF_INTERNAL;

  di->active_io_count++;
  nodep->num_active_io++;
  FCSTATCTR.fcpCmd++;

  issue_iocb_cmd(binfo, rp, 0);
  return;
}
/****************************************/
/* Print Format Declarations Start Here */
/****************************************/
_local_ int  fc_sprintf_fargs( uchar *string, void *control, char *fixArgs);

#define  LENGTH_LINE 71
#define  MAX_IO_SIZE 32 * 2            /* iobuf cache size */
#define  MAX_TBUFF   18 * 2            /* temp buffer size */

typedef union {                        /* Pointer to table of arguments. */
   ulong        *ip;
   ulong        *lip;
   ulong        *uip;
   ulong        *luip;
   ulong       **luipp;
   uchar        *cp;
   uchar       **csp;
} ARGLIST; 

typedef struct {
   uchar *string;
   long   index;
   int    count;
   uchar  buf[MAX_IO_SIZE + MAX_TBUFF];  /* extra room to convert numbers */
} PRINTBLK;

/*
 * ASCII string declarations
 */
static char  dig[]            = {"0123456789ABCDEF"};
static char  ds_disabled[]    = "disabled";
static char  ds_enabled[]     = "enabled";
static char  ds_none[]        = "none";
static char  ds_null_string[] = "";
static char  ds_unknown[]     = "unknown";

/*
 * Function Declarations
 */
_local_ int  add_char( PRINTBLK * io, uchar ch);
_local_ int  add_string( PRINTBLK * io, uchar * string);
_local_ int  fmtout( uchar *ostr, uchar *control, va_list inarg);
_local_ void print_string( PRINTBLK * io);
_local_ int  long_itos( long val, uchar * cp, long base);


/**********************************************************/
/** expanded_len                                          */
/**   determine the length of the string after expanding  */
/**********************************************************/
_local_
int expanded_len(
uchar *sp)
{
   register int  i;
   uchar         c;

   i = 0;
   while ((c = *sp++) != 0) {
      if (c < 0x1b) {
         if ((c == '\r') || (c == '\n'))
            break;  /* stop at cr or lf */
         i++; /* double it */
      }
   i++;
   }
   return (i);
} /* expanded_len */

/*************************************************/
/**  long_itos                                  **/
/**    Convert long integer to decimal string.  **/
/**    Returns the string length.               **/
/*************************************************/
_local_
int long_itos(
long    val,       /* Number to convert. */
uchar * cp,        /* Store the string here. */
long    base)      /* Conversion base. */
{
   uchar  tempc[16];
   uchar *tcp;
   int    n=0;     /* number of characters in result */
   ulong  uval;    /* unsigned value */
   
   *(tcp=(tempc+15))=0;
   if (base<0) {
      /* needs signed conversion */
      base= -base;
      if (val<0) {
         n=1;
         val = -val;
      }
      do {
         *(--tcp)=dig[ (int)(val%base)];
         val /= base;
      } while (val);
   }
   else {
      uval=val;
      do {
         *(--tcp)=dig[ (int)(uval%base)];
         uval/=base;
      } while(uval);
   }
   if (n)
      *(--tcp)='-';
   n=(int)((long)&tempc[15] - (long)tcp);
   fc_bcopy( tcp, cp, n+1);   /* from, to, cnt */
   return(n);
} /* long_itos */

/****************************************/
/**  add_char                          **/
/****************************************/
_local_
int add_char(
PRINTBLK      * io,
uchar   ch)
{
   int index;

   if (ch < 0x1b) {
      switch (ch) {
         case 0xd: /* carriage return */
            io->count = -1;   /* will be incremented to 0, below */
            break;
         case 0x8: /* back space */
            io->count -= 2;   /* will be incremented to 1 less, below */
            break;
         case 0xa:   /* line feed */
         case 0x7: /* bell */
         case 0x9: /* hortizontal tab */
         case 0xe: /* shift out */
         case 0xf: /* shift in */
            io->count--;         /* will be incremented to same, below */
            break;
         default:
            add_char(io, '^');   
            ch |= 0x40;
            break;
      }
   }
   io->count++;
   if (io->string != NULL) {
      *io->string = ch;
      *++io->string = '\0';
      return (0);
   }
      
   index = io->index;
   if( index < (MAX_IO_SIZE + MAX_TBUFF -2)) {
      io->buf[index] = ch;
      io->buf[++index] = '\0';
   }
   return (++io->index);
} /* add_char */

/****************************************/
/**  add_string                        **/
/****************************************/
_local_
int add_string(
PRINTBLK * io,
uchar    * string)
{
   if (io->string != NULL) {
      io->string = 
           (uchar *)fc_strcpy( (char *)io->string, (char *)string); /* dst, src */
      return (0);
   }
   return (io->index = ((long)(fc_strcpy( (char *)&io->buf[io->index],
            (char *)string))) - ((long)((char *)io->buf)));  /* dst, src */
} /* add_string */

/*****************************************************/
/**  print_string                                   **/
/**    takes print defn, prints data, zeroes index  **/
/*****************************************************/
_local_
void print_string(
PRINTBLK * io)
{
   io->index = 0;
   fc_print( (char *)&io->buf[0],0,0); 
} /* print_string */

/*VARARGS*/
/*****************************************/
/**  fmtout                             **/
/**    Low-level string print routines. **/
/*****************************************/
_local_
int fmtout (
uchar      *ostr,              /* Output buffer, or NULL if temp */
uchar      *control,           /* Control string */
va_list     inarg)             /* Argument list */
{
   short         temp;            /* Output channel number if string NULL. */
   int           leftadj;         /* Negative pararameter width specified. */
   int           longflag;        /* Integer is long. */
   int           box = FALSE;     /* not from body */
   int           chr;             /* control string character */
   uchar         padchar;         /* Pad character, typically space. */
   int           width;           /* Width of subfield. */
   int           length;          /* Length of subfield. */
   uchar        *altctl;          /* temp control string */
   ARGLIST       altarg;
   ARGLIST       arg;
   PRINTBLK      io;

   union {                        /* Accumulate parameter value here. */
      uint16 tlong;
      uint16 tulong;
      long   ltlong;
      ulong  ltulong;
      uchar  str[4];
      uint16 twds[2];
   } lw;
   
   union {                        /* Used by case %c */
      int    intchar;
      uchar  chr[2];
   } ichar;
   
   arg.uip = (ulong *)inarg;
   io.index = 0;
   io.count = 0;

   if( (io.string = ostr) != (uchar *)NULL)
      *ostr = 0;                  /* initialize output string to null */
   control--;

   mainloop:
   altctl = NULL;             
   
   while ((length = *++control) != 0)
      { /* while more in control string */
      if (length !='%') {         /* format control */
         if ((length == '\n') && box) {
            fc_print( (char *)&io.buf[0],0,0); 
            continue;
         }
         if (add_char( &io, (uchar) length) >= MAX_IO_SIZE)
            print_string(&io); /* write it */
         continue;
      } 
      leftadj = (*++control == '-');
      if (leftadj)
         ++control;
      padchar = ' ';
      width = 0;
      if ((uint16)(length = (*control - '0')) <= 9) {
         if (length == 0)
            padchar = '0';
         width = length;
         while ((uint16)(length = (*++control - '0')) <= 9 )
            width = width*10+length;
      }
      longflag = ( *control == 'l');
      if ( longflag)
         ++control;

      chr = (int)(*control);
      if( chr != 'E') {
         chr |= 0x20;
      }

      switch (chr) { 
         case 'a':
            longflag = 1;
            temp=16;
            padchar = '0';
            length = width = 8;
            goto nosign;
         case 'b':
            temp=2;
            goto nosign;
         case 'o':
            temp=8;
            goto nosign;
         case 'u':
            temp=10;
            goto nosign;
         case 'x':
            temp=16;
            goto nosign;
   
         case 'e':
            ostr = (uchar *)va_arg(inarg, char *);
            if ((chr == 'e') && 
                 ((*(long *)ostr) == (long)NULL) &&
                   ((*(uint16 *)&ostr[4]) == (uint16)0)) {
               ostr = (uchar *)ds_unknown;
               length = 7;
               break;
            }
            temp = -1;
            length = MAX_IO_SIZE -1;
            fc_strcpy((char *)&io.buf[MAX_IO_SIZE],
                           "00-00-00-00-00-00");  /* dst, src */
            do {
               long_itos((long)( ostr[++temp] + 256), lw.str, 16);
               io.buf[++length] = lw.str[1];
               io.buf[++length] = lw.str[2];
            } while (++length < MAX_IO_SIZE+17);
            ostr = &io.buf[MAX_IO_SIZE];
            length = 17;
            break;
   
         case 'E':
            ostr = (uchar *)va_arg(inarg, char *);
            if ((chr == 'E') && 
                 ((*(long *)ostr) == (long)NULL) &&
                   ((*(long *)&ostr[4]) == (long)NULL)) {
               ostr = (uchar *)ds_unknown;
               length = 7;
               break;
            }
            temp = -1;
            length = MAX_IO_SIZE -1;
            fc_strcpy( (char *)&io.buf[MAX_IO_SIZE],
                            "00-00-00-00-00-00-00-00");  /* dst, src */
            do {
               long_itos((long)( ostr[++temp] + 256), lw.str, 16);
               io.buf[++length] = lw.str[1];
               io.buf[++length] = lw.str[2];
            } while (++length < MAX_IO_SIZE+23);
            ostr = &io.buf[MAX_IO_SIZE];
            length = 23;
            break;
 
          case 'f':    /* flags */
            ostr = (uchar *)ds_disabled;
            length = 8;
            if (va_arg(inarg, char *) != 0) {     /* test value */
               ostr = (uchar *)ds_enabled;
               length = 7;
            }
            if (chr == 'F') { 
               length -= 7;
               ostr = (uchar *)"-";
            }
            break;
   
         case 'i':
            ostr = (uchar *)va_arg(inarg, char *);
            if ((chr == 'i') && *(long *)ostr == (long)NULL)
               goto putnone;
            temp = 0;
            length = MAX_IO_SIZE;
            do {
               length += long_itos((long) ostr[temp], &io.buf[length], 10);
               if ( ++temp >= 4)
                  break;
               io.buf[length] = '.';
               length++;
            } while (TRUE);
            ostr = &io.buf[MAX_IO_SIZE];
            length -= MAX_IO_SIZE;
            break;
   
         case 'y':    /* flags */
            if ( va_arg(inarg, char *) != 0) {     /* test value */
               ostr = (uchar*)"yes";
               length = 3;
            }
            else {
               ostr = (uchar*)"no";
               length = 2;
            }
            break;

         case 'c':
            if (chr == 'C') { /* normal, control, or none */
               if ((length = va_arg(inarg, int)) < ' ') {
                  if (length == 0) {
                     ostr = (uchar *)ds_none;
                     length = 4;
                  }
                  else {
                     io.buf[MAX_IO_SIZE]   = '^';
                     io.buf[MAX_IO_SIZE+1] = ((uchar)length) + '@';
                     io.buf[MAX_IO_SIZE+2] = 0;
                     ostr = &io.buf[MAX_IO_SIZE];
                     length = 2;
                  }
                  arg.ip++;
                  break;
               }
            } /* normal, control, or none */
            
            ichar.intchar = va_arg(inarg, int);
            ostr = &ichar.chr[0];
            length=1;
            break;
   
         case 'd':
            temp = -10;
   nosign:
            if (longflag)
               lw.ltulong = va_arg(inarg, ulong);
            else if (temp < 0)
               lw.ltlong  = va_arg(inarg, long);
            else
               lw.ltulong = va_arg(inarg, ulong);
/*
   nosign2:
*/
            length = long_itos( lw.ltlong, ostr = &io.buf[MAX_IO_SIZE], temp);
            break;
   
         case 's':
            ostr = (uchar *)va_arg(inarg, char *);     /* string */
            if ((chr == 's') || (*ostr != '\0')) {
               length = expanded_len(ostr);
               break;
            }
   putnone:
            ostr = (uchar *)ds_none;
            length = 4;
            break;
   
         case 't':                  /* tabbing */
            if ((width -= io.count) < 0) /* Spaces required to get to column. */
               width = 0;
            length = 0;             /* nothing other than width padding. */
            ostr = (uchar *)ds_null_string;
            break;
         case ' ':
            width = va_arg(inarg, int);
            length = 0;             /* nothing other than width padding. */
            ostr = (uchar *)ds_null_string;
            break;
   
         default:
            ostr=control;
            length=1;
            break;
      } /* switch on control */
   
      if (length < 0) { /* non printing */
         if (add_string( &io, ostr) >= MAX_IO_SIZE)
            print_string(&io);    /* no more room, dump current buffer */
         continue;
      } /* non printing */
   
   
      if (!leftadj && width > length) {
         while (--width >= length) {
            if (add_char( &io, padchar) >= MAX_IO_SIZE)
               print_string(&io); /* write it */
         }
      }
   
      if (width>length)
         width -= length;
      else
         width = 0;
   
      if (length <= 1) {
         if (length == 1) {
            if (add_char( &io, *ostr) >= MAX_IO_SIZE)
               print_string(&io); /* write it */
         }
      }
      else {
         while ((temp = *ostr++) != 0) {
            if (add_char( &io, (uchar) temp) >= MAX_IO_SIZE)
               print_string(&io); /* write it */
         }
      }
   
      while (--width >= 0) {
         if (add_char( &io, padchar) >= MAX_IO_SIZE)
            print_string(&io); /* write it */
      }
   
   } /* while more in control string */
   
   if (altctl != NULL) {
      control = altctl;
      arg.ip  = altarg.ip;
      goto mainloop;
   }
   
   if (io.index)                       /* anything left? */
      print_string(&io);               /* write it */
   
   return(io.count);
} /* fmtout */
/*FIXARGS*/
_local_ int 
fc_sprintf_fargs(
uchar *string,                     /* output buffer */
void  *control,                    /* format string */
char  *fixArgs)                    /* control arguments */
{
   return( fmtout((uchar *)string, (uchar *)control, fixArgs));
} /* fc_sprintf_fargs */
/*VARARGS*/
int fc_sprintf_vargs(
   void *string,                     /* output buffer */
   void *control,                    /* format string */
   ...)                              /* control arguments */
{
   int iocnt;
   va_list args;
   va_start(args, control);

   iocnt = fmtout((uchar *)string, (uchar *)control, args);
   va_end( args);
   return( iocnt);
} /* fc_sprintf_vargs */
/****************************************/
/**  fc_log_printf_msg_vargs           **/
/****************************************/
/*
All log messages come through this routine.
All log messages are unique.
All log messages are define by a msgLogDef messages structure. 
*/
/*VARARGS*/
_static_ int 
fc_log_printf_msg_vargs(
   int         brdno,
   msgLogDef * msg,          /* Pionter to LOG msg structure */
   void      * control,
   ...)
{
   uchar str2[MAX_IO_SIZE + MAX_TBUFF];  /* extra room to convert numbers */
   int iocnt;
   int log_only;
   va_list args;
   va_start(args, control);
  
   log_only = 0;
   if( fc_check_this_log_msg_disabled( brdno, msg, &log_only))
      return(0); /* This LOG message disabled */

   /* 
   If LOG message is disabled via any SW method, we SHOULD NOT get this far!
   We should have taken the above return. 
   */

   str2[0] = '\0';
   iocnt = fc_sprintf_fargs(str2, control, args);
   va_end( args);

   return( log_printf_msgblk( brdno, msg, (char *)str2, log_only));
} /* fc_log_printf_msg_vargs */

/*****************************************************/
/** Function name : fc_check_this_log_msg_disabled  **/
/**                                                 **/
/** Description   :                                 **/
/**                                                 **/
/** Return        : 0  LOG message enabled          **/
/**               : 1  LOG message disabled         **/
/*****************************************************/
int fc_check_this_log_msg_disabled( int brdno,
                                    msgLogDef *msg,
                                    int *log_only)
{
   fc_dev_ctl_t  * p_dev_ctl;
   iCfgParam     * clp;
   int             verbose;

   verbose = 0;
   if( msg->msgOutput == FC_MSG_OPUT_DISA)
      return(1);        /* This LOG message disabled */

   if ((p_dev_ctl = DD_CTL.p_dev[brdno])) {
      clp = DD_CTL.p_config[brdno];
      if((*log_only = clp[CFG_LOG_ONLY].a_current) > 1)
         return(1);     /* This LOG message disabled */
      verbose = clp[CFG_LOG_VERBOSE].a_current;
   }

   if( msg->msgOutput == FC_MSG_OPUT_FORCE)
      return(0);        /* This LOG message enabled */
   /*
    * If this is a verbose message (INFO or WARN) and we are not in 
    * verbose mode, return 1. If it is a verbose message and the verbose 
    * error doesn't match our verbose mask, return 1.
    */
   if( (msg->msgType == FC_LOG_MSG_TYPE_INFO) || 
         (msg->msgType == FC_LOG_MSG_TYPE_WARN)) {
      /* LOG msg is INFO or WARN */
      if ((msg->msgMask & verbose) == 0)
         return(1);     /* This LOG mesaage disabled */
   }
   return(0);           /* This LOG message enabled */
} /* fc_check_this_log_msg_disabled */

/*************************************************/
/**   fc_asc_to_hex                             **/
/**   Convert an ASCII hex character to hex.    **/
/**   Return  Hex value if success              **/
/**          -1 if character not ASCII hex      **/
/*************************************************/

_forward_ int 
fc_asc_to_hex(
   uchar c)     /* Character to convert */
{
if (c >= '0' && c <= '9') 
   return(c - '0');
else if (c >= 'A' && c <= 'F')
   return(c - 'A'+ 10);
else if (c >= 'a' && c <= 'f')
   return(c - 'a'+ 10);
else
   return(-1);
} /* fc_asc_to_hex */

/***************************************************/
/**  fc_asc_seq_to_hex                            **/
/**                                               **/
/**  Convert an ASCII character sequence to a     **/
/**  hex number sequence                          **/
/**                                               **/
/**  return >0  Success. Return number of ASCII   **/
/**             hex characters converted.         **/
/**         -1  Input byte count < 1              **/
/**         -2  Input byte count > max            **/
/**         -3  Output buffer to small            **/
/**         -4  Input character sequence not      **/
/**             ASCII hex.                        **/
/***************************************************/

/*
This routine converts an ASCII char stream of byte into
a stream of hex bytes. The byte order of the input and
output stream are identical. The caller must deal with
SWAPPING bytes if required.

The maximum number of ASCII hex characters that can be 
convert to hex is hard coded by the LOCAL define 
MAX_ASC_HEX_CHARS_INPUT. 

Two ASCII hex input characters require 1 byte of output 
buffer.

A NULL terminator at the end of an ASCII hex input string 
is not required nor is it counted in the strings byte size.

To determine the byte size of the output buffer:
(1) Add 1 to input buffer byte size if size is odd.
(2) Output buffer size = input buffer size / 2.
    
Therefore an input buffer containing 10 ASC hex chars 
requires an output buffer size of 5 bytes.

An input buffer containing 11 ASC hex chars requires an 
output buffer size of 6 bytes.
*/

_forward_ int
fc_asc_seq_to_hex( fc_dev_ctl_t *p_dev_ctl,
                   int     input_bc,   /* Number of bytes (ASC hex chars) to be converted */
                   int     output_bc,  /* Number of bytes in hex output buffer (modulo INT) */
                   char   *inp,        /* Pointer to ASC hex input character sequence */
                   char   *outp)       /* Pointer to hex output buffer */
{
#define HEX_DIGITS_PER_BYTE        2
#define MAX_ASC_HEX_CHARS_INPUT   32      /* Limit damage if over-write */
#define MAX_BUF_SIZE_HEX_OUTPUT   (MAX_ASC_HEX_CHARS_INPUT / HEX_DIGITS_PER_BYTE)

   FC_BRD_INFO  *binfo;
   int           lowNib, hiNib;
   int           inputCharsConverted;
   uchar         twoHexDig;

   binfo = &BINFO;
   inputCharsConverted = 0;
   lowNib = -1;
   hiNib  = -1;

   if(input_bc < 1) {
      /* Convert ASC to hex. Input byte cnt < 1. */ 
      fc_log_printf_msg_vargs( binfo->fc_brd_no,
             &fc_msgBlk1210,                     /* ptr to msg structure */
              fc_mes1210,                        /* ptr to msg */
               fc_msgBlk1210.msgPreambleStr);    /* begin & end varargs */
      return(-1);
   }
   if(input_bc > MAX_ASC_HEX_CHARS_INPUT) {
      /* Convert ASC to hex. Input byte cnt > max <num> */ 
      fc_log_printf_msg_vargs( binfo->fc_brd_no,
             &fc_msgBlk1211,                     /* ptr to msg structure */
              fc_mes1211,                        /* ptr to msg */
               fc_msgBlk1211.msgPreambleStr,     /* begin varargs */
                MAX_ASC_HEX_CHARS_INPUT);        /* end varargs */
      return(-2);
   }
   if((output_bc * 2) < input_bc) {
      /* Convert ASC to hex. Output buffer to small. */ 
      fc_log_printf_msg_vargs( binfo->fc_brd_no,
             &fc_msgBlk1212,                     /* ptr to msg structure */
              fc_mes1212,                        /* ptr to msg */
               fc_msgBlk1212.msgPreambleStr);    /* begin & end varargs */
      return(-4);
   }

   while( input_bc) {
      twoHexDig = 0;
      lowNib = -1;
      hiNib = fc_asc_to_hex( *inp++);
      if( --input_bc > 0) {
         lowNib = fc_asc_to_hex( *inp++);
         input_bc--;
      }
      if ((lowNib < 0) || (hiNib < 0)) {
         /* Convert ASC to hex. Input char seq not ASC hex. */ 
         fc_log_printf_msg_vargs( binfo->fc_brd_no,
                &fc_msgBlk1213,                     /* ptr to msg structure */
                 fc_mes1213,                        /* ptr to msg */
                  fc_msgBlk1213.msgPreambleStr);    /* begin & end varargs */
         return( -4);
      }
      if( lowNib >= 0) {
         /* There were 2 digits */
         hiNib <<= 4;
         twoHexDig = (hiNib | lowNib);
         inputCharsConverted += 2;
      }
      else {
         /* There was a single digit */
         twoHexDig = lowNib;
         inputCharsConverted++;
      }                 
      *outp++ = twoHexDig;
   } /* while */
   return(inputCharsConverted); /* ASC to hex conversion complete. Return # of chars converted */
} /* fc_asc_seq_to_hex */

/********************************************/
/** fc_is_digit                            **/
/**                                        **/
/** Check if ASCII input value is numeric. **/
/**                                        **/
/** Return 0 = input NOT numeric           **/
/**        1 = input IS numeric            **/
/********************************************/
_forward_ int
fc_is_digit(int chr)
{
   if( (chr >= '0') && (chr <= '9'))
      return(1);
   return(0);
} /* fc_is_digit */

