/*
 * touchpad.c -- code to enable/disable touchpad
 * 
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the
 * Free Software Foundation; either version 2, or (at your option) any
 * later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * Written by Sos Pter <sp@osb.hu>, 2002-2003
 */


#include <linux/config.h>
#include <linux/module.h>
#include <linux/version.h>
#include <linux/types.h>
#include <linux/init.h>
#include <linux/pm.h>
#include <linux/proc_fs.h>

#include <asm/system.h>
#include <asm/errno.h>
#include <asm/io.h>
#include <asm/uaccess.h>

#ifdef OMNIBOOK_STANDALONE
#include "omnibook.h"
#else
#include <linux/omnibook.h>
#endif

#include "ec.h"

static struct pm_dev *pm_touchpad;
static pm_callback pm_touchpad_callback = NULL;

static struct proc_dir_entry *proc_touchpad;


#ifdef CONFIG_PSMOUSE
/* Touchpad is enabled by default */
int omnibook_touchpad_enabled = 1;
#else
int omnibook_touchpad_enabled = 0;
#endif

static int omnibook_touchpad_on(void)
{
	if (omnibook_kbc_command(OMNIBOOK_KBC_CONTROL_CMD, OMNIBOOK_KBC_CMD_TOUCHPAD_ENABLE)) {
		printk(KERN_ERR "%s: failed touchpad enable command.\n", MODULE_NAME);
		return -EIO;
	}
	return 0;
}

static int omnibook_touchpad_off(void)
{
	if (omnibook_kbc_command(OMNIBOOK_KBC_CONTROL_CMD, OMNIBOOK_KBC_CMD_TOUCHPAD_DISABLE)) {
		printk(KERN_ERR "%s: failed touchpad disable command.\n", MODULE_NAME);
		return -EIO;
	}
	return 0;
}

/*
 * Power management handler: on resume it redisables the touchpad if it was disabled previously
 */

static int pm_touchpad_handler(struct pm_dev *dev, pm_request_t rqst, void *data)
{
	switch (rqst) {
	case PM_RESUME:
		if (! omnibook_touchpad_enabled)
			return omnibook_touchpad_off();
		break;
	}
	return 0;
}

static int omnibook_touchpad_register(void)
{
		pm_touchpad_callback = pm_touchpad_handler;
		pm_touchpad = pm_register(PM_SYS_DEV, PM_SYS_KBC, pm_touchpad_callback);
		if (pm_touchpad)
			return 0;
		else
			return -EFAULT;
}

static void omnibook_touchpad_unregister(void)
{
		pm_touchpad_callback = NULL;
		pm_unregister(pm_touchpad);
}

int omnibook_touchpad_enable(void)
{
#ifdef CONFIG_PSMOUSE
	switch (omnibook_ectype) {
	case XE3GF:
	case XE3GC:
		if (! omnibook_touchpad_enabled) {
			if (omnibook_touchpad_on())
				return -EIO;
			omnibook_touchpad_enabled = 1;
			printk(KERN_INFO "%s: Touchpad is enabled.\n", MODULE_NAME);
		}
		break;
	case OB500:
	case OB510:
		omnibook_touchpad_enabled = 0;
		return -ENODEV;
		break;
	default:
		omnibook_touchpad_enabled = 1;
		return -ENODEV;
	}
#endif
	return 0;
}

int omnibook_touchpad_disable(void)
{
#ifdef CONFIG_PSMOUSE
	switch (omnibook_ectype) {
	case XE3GF:
	case XE3GC:
		if (omnibook_touchpad_enabled) {
			if (omnibook_touchpad_off()) {
				return -EIO;
			}
			omnibook_touchpad_enabled = 0;
			printk(KERN_INFO "%s: Touchpad is disabled.\n", MODULE_NAME);
		}
		break;
	case OB500:
	case OB510:
		omnibook_touchpad_enabled = 0;
		return -ENODEV;
		break;
	default:
		omnibook_touchpad_enabled = 1;
		return -ENODEV;
	}
#endif
	return 0;
}

static int omnibook_touchpad_status(char *buffer, char **start, off_t off, int count, int *eof, void *data)
{
	int len;
	char *b = buffer;
	char *str;

	str = (omnibook_touchpad_enabled) ? "enabled" : "disabled";

        b += sprintf(b, "Touchpad is %s\n", str);
	
	len = b - buffer;
	if (len < off + count)
		*eof = 1;
	*start = buffer + off;
	len -= off;
	if (len > count)
		len = count;
	if (len < 0)
		len = 0;
	return len;
}

static int omnibook_touchpad_set(struct file *file, const char *buffer, unsigned long count, void *data)
{
	char status[1] = {'\0'};

	if (copy_from_user(status, buffer, 1))
		return -EFAULT;
	switch (*status) {
	case '0':
		omnibook_touchpad_disable();
		break;
	case '1':
		omnibook_touchpad_enable();
		break;
	default:
		count = -EINVAL;
	}
	return count;
}

int __init omnibook_touchpad_init(void)
{
#ifdef CONFIG_PSMOUSE
	int retval;
	mode_t pmode;
	
	switch (omnibook_ectype) {
	case XE3GF:
	case XE3GC:
		pmode = S_IFREG | S_IWUSR | S_IRUGO;
		if (omnibook_userset)
			pmode = pmode | S_IWUGO;
		proc_touchpad = create_proc_entry("touchpad", pmode, omnibook_proc_root);
		break;
	default:
		printk(KERN_INFO "%s: Touchpad handling is unsupported on this machine.\n", MODULE_NAME);
		omnibook_touchpad_enabled = 1;
		return 0;
	}
	if (proc_touchpad) {
		proc_touchpad->read_proc = omnibook_touchpad_status;
		proc_touchpad->write_proc = omnibook_touchpad_set;
	} else {
		printk(KERN_ERR "%s: Unable to create /proc/%s/touchpad.\n", MODULE_NAME, MODULE_NAME);
		return -ENOENT;
	}
	retval = omnibook_touchpad_register();
	if (retval)
		return retval;

	return 0;
#else
	printk(KERN_ERR "%s: PS/2 mouse support is not compiled into your kernel.\n", MODULE_NAME);
	return -ENODEV;
#endif
}

void __exit omnibook_touchpad_cleanup(void)
{
#ifdef CONFIG_PSMOUSE
	if (proc_touchpad)
		remove_proc_entry("touchpad", omnibook_proc_root);
	if (pm_touchpad)
		omnibook_touchpad_unregister();
	omnibook_touchpad_enable();
#endif
}

EXPORT_SYMBOL(omnibook_touchpad_enabled);
EXPORT_SYMBOL(omnibook_touchpad_enable);
EXPORT_SYMBOL(omnibook_touchpad_disable);

/* End of file */
